#!/usr/bin/env python3
import os
import sys
import subprocess
import glob
from datetime import datetime

# =================================================================
# KONFIGURASJON
# =================================================================
# Sett til True for å slette lydfiler etter at de er lagret i databasen
DELETE_AFTER_PROCESSING = True 
ENV_PATH = "/home/pi/key_info.env"
RECORD_DIR = "/home/pi/recordings"
# Timen (0-23) der AI-transkribering skal kjøre
TRANSCRIPTION_HOUR = 3 

def fast_dependency_check():
    """Sjekker biblioteker lynraskt uten å starte pip unødvendig"""
    dependencies = {
        "whisper": "openai-whisper",
        "mysql.connector": "mysql-connector-python",
        "dotenv": "python-dotenv",
        "psutil": "psutil"
    }
    
    missing = []
    for module, package in dependencies.items():
        try:
            __import__(module)
        except ImportError:
            missing.append(package)
    
    if missing:
        print(f"Installerer manglende biblioteker: {missing}")
        for pkg in missing:
            subprocess.run([sys.executable, "-m", "pip", "install", pkg, "--break-system-packages"], check=True)

# Kjør lynrask sjekk før vi importerer de store bibliotekene
fast_dependency_check()

import mysql.connector
import whisper
import psutil
from dotenv import load_dotenv

def get_optimal_config():
    """Velger optimal Whisper-modell basert på tilgjengelig RAM"""
    total_ram = psutil.virtual_memory().total / (1024**3)
    
    try:
        with open("/proc/device-tree/model", "r") as f:
            model_info = f.read().strip()
    except:
        model_info = "Ukjent maskinvare"

    print(f"\n--- Systemanalyse ---")
    print(f"Maskinvare: {model_info}")
    print(f"Tilgjengelig RAM: {total_ram:.2f} GB")

    if total_ram < 1.5:
        chosen_model = "tiny"
    elif total_ram < 3.5:
        chosen_model = "base"
    elif total_ram < 7.5:
        chosen_model = "small"
    elif total_ram < 15.0:
        chosen_model = "medium"
    else:
        chosen_model = "large"

    print(f"Optimal konfigurasjon valgt: '{chosen_model}'")
    return chosen_model

def get_db_connection():
    if not os.path.exists(ENV_PATH):
        print(f"🚨 Fant ikke miljøfilen {ENV_PATH}!")
        sys.exit(1)
    
    load_dotenv(ENV_PATH)
    return mysql.connector.connect(
        host=os.getenv("DB_HOST", "localhost"),
        user=os.getenv("DB_USER"),
        password=os.getenv("DB_PASSWORD"),
        database=os.getenv("DB_NAME"),
        charset='utf8mb4',
        collation='utf8mb4_unicode_ci'
    )

def archive_files():
    # 1. Hent alle filer og rens ut tomme/små filer (<10KB) umiddelbart
    all_raw_files = glob.glob(f"{RECORD_DIR}/*.wav")
    valid_files = []
    deleted_small_files = 0

    for f_path in all_raw_files:
        if os.path.getsize(f_path) < 10240: # 10 KB grense
            os.remove(f_path)
            deleted_small_files += 1
        else:
            valid_files.append(f_path)

    if deleted_small_files > 0:
        print(f"🧹 Renset ut {deleted_small_files} tomme/små filer (< 10KB).")

    # 2. Tidssjekk: Skal vi kjøre transkribering nå?
    current_hour = datetime.now().hour
    if current_hour != TRANSCRIPTION_HOUR:
        print(f"⏰ Klokken er {current_hour}:00. Hopper over transkribering til kl. 0{TRANSCRIPTION_HOUR}:00.")
        return

    # --- Herfra kjører koden kun kl. 03:00 ---
    if not valid_files:
        print("Ingen relevante lydfiler funnet for transkribering.")
        return

    # 3. Last modell og koble til DB
    model_name = get_optimal_config()
    print(f"Laster AI-modellen '{model_name}' inn i minnet...")
    model = whisper.load_model(model_name)
    
    conn = get_db_connection()
    cursor = conn.cursor()

    print(f"\nStarter nattlig prosessering av {len(valid_files)} filer...")

    for file_path in valid_files:
        filename = os.path.basename(file_path)
        
        try:
            # 4. Transkribering
            result = model.transcribe(file_path, language="no", fp16=False)
            tekst = result['text'].strip()
            
            if not tekst: tekst = "[Ingen tale detektert]"

            # 5. Metadata og SQL
            parts = filename.replace(".wav", "").split("_")
            timestamp_str = f"{parts[0]} {parts[1]}"
            timestamp = datetime.strptime(timestamp_str, '%Y%m%d %H%M%S')
            freq = f"{parts[2]}.{parts[3]}"

            query = "INSERT INTO radio_archive (timestamp, frekvens, tekst, filnavn) VALUES (%s, %s, %s, %s)"
            cursor.execute(query, (timestamp, freq, tekst, filename))
            conn.commit()
            
            print(f"✅ Fullført: {filename} -> {tekst[:50]}...")
            
            if DELETE_AFTER_PROCESSING:
                os.remove(file_path)
            
        except Exception as e:
            print(f"❌ Feil ved prosessering av {filename}: {e}. Sletter korrupt fil.")
            # Sletter filen selv om den feiler, slik at den ikke fyller opp disken permanent
            if os.path.exists(file_path):
                os.remove(file_path)

    cursor.close()
    conn.close()
    print("\n--- Nattlig arkivering ferdig ---")

if __name__ == "__main__":
    archive_files()
