#!/usr/bin/env python3
import os
import sys
import subprocess
import glob
import wave
import contextlib
from datetime import datetime

# =================================================================
# KONFIGURASJON
# =================================================================
DELETE_AFTER_PROCESSING = True
ENV_PATH = "/home/pi/key_info.env"
RECORD_DIR = "/home/pi/radio_audio/recordings"
TRANSCRIPTION_HOUR = 3

def fast_dependency_check():
    """Sjekker biblioteker lynraskt"""
    dependencies = {
        "whisper": "openai-whisper",
        "mysql.connector": "mysql-connector-python",
        "dotenv": "python-dotenv",
        "psutil": "psutil"
    }
    missing = []
    for module, package in dependencies.items():
        try:
            __import__(module)
        except ImportError:
            missing.append(package)
    if missing:
        for pkg in missing:
            subprocess.run([sys.executable, "-m", "pip", "install", pkg, "--break-system-packages"], check=True)

fast_dependency_check()

import mysql.connector
import whisper
import psutil
from dotenv import load_dotenv

def get_optimal_config():
    total_ram = psutil.virtual_memory().total / (1024**3)
    if total_ram < 3.5: chosen_model = "base"
    elif total_ram < 7.5: chosen_model = "small"
    elif total_ram < 15.0: chosen_model = "medium"
    else: chosen_model = "large"
    return chosen_model

def get_db_connection():
    load_dotenv(ENV_PATH)
    return mysql.connector.connect(
        host=os.getenv("DB_HOST", "localhost"),
        user=os.getenv("DB_USER"),
        password=os.getenv("DB_PASSWORD"),
        database=os.getenv("DB_NAME"),
        charset='utf8mb4',
        collation='utf8mb4_unicode_ci'
    )

def get_duration(file_path):
    """Henter varighet i sekunder fra wav-filen"""
    try:
        with contextlib.closing(wave.open(file_path, 'r')) as f:
            frames = f.getnframes()
            rate = f.getframerate()
            return int(frames / float(rate))
    except:
        return 0

def archive_files():
    if not os.path.exists(RECORD_DIR): return

    all_raw_files = glob.glob(f"{RECORD_DIR}/*.wav")
    valid_files = []
    
    # 1. Rens ut småfiler hver time
    for f_path in all_raw_files:
        if os.path.getsize(f_path) < 10240:
            os.remove(f_path)
        else:
            valid_files.append(f_path)

    # 2. Tidssjekk for transkribering
    current_hour = datetime.now().hour
    if current_hour != TRANSCRIPTION_HOUR:
        print(f"⏰ Klokken er {current_hour}:00. Småfiler slettet. AI kjører kl. 03:00.")
        return

    if not valid_files: return

    model = whisper.load_model(get_optimal_config())
    conn = get_db_connection()
    cursor = conn.cursor()

    print(f"[*] Starter nattlig prosessering av {len(valid_files)} filer...")

    for file_path in valid_files:
        filename = os.path.basename(file_path)
        try:
            # Hent varighet
            duration = get_duration(file_path)

            # Transkribering
            result = model.transcribe(file_path, language="no", fp16=False)
            tekst = result['text'].strip()
            if not tekst: tekst = "[Ingen tale detektert]"

            # Metadata-mapping til din tabellstruktur
            parts = filename.replace(".wav", "").split("_")
            timestamp_str = f"{parts[1]} {parts[2]}"
            timestamp = datetime.strptime(timestamp_str, '%Y%m%d %H%M%S')
            
            frekvens_val = "VHF-SCAN"
            navn_val = "Maritim Scan"

            # SQL-spørring tilpasset dine kolonnenavn
            query = """
            INSERT INTO radio_archive (timestamp, frekvens, navn, tekst, varighet, filnavn) 
            VALUES (%s, %s, %s, %s, %s, %s)
            """
            cursor.execute(query, (timestamp, frekvens_val, navn_val, tekst, duration, filename))
            conn.commit()

            print(f"✅ Arkivert: {filename} ({duration}s)")
            if DELETE_AFTER_PROCESSING:
                os.remove(file_path)

        except Exception as e:
            print(f"❌ Feil ved {filename}: {e}")
            if os.path.exists(file_path): os.remove(file_path)

    cursor.close()
    conn.close()

if __name__ == "__main__":
    archive_files()
