#!/usr/bin/env python3
import os
import sys
import subprocess
import glob
import wave
import contextlib
from datetime import datetime

import torch
torch.set_num_threads(2)

import mysql.connector
import whisper
import psutil
from dotenv import load_dotenv


# =================================================================
# KONFIGURASJON
# =================================================================
DELETE_AFTER_PROCESSING = True
ENV_PATH = "/home/pi/key_info.env"
RECORD_DIR = "/home/pi/radio_audio/recordings"
TRANSCRIPTION_HOUR = 3  # Når AI skal kjøre automatisk

JUMP_OVER_MD = 200

def fast_dependency_check():
    """Sjekker biblioteker lynraskt"""
    dependencies = {
        "whisper": "openai-whisper",
        "mysql.connector": "mysql-connector-python",
        "dotenv": "python-dotenv",
        "psutil": "psutil"
    }
    missing = []
    for module, package in dependencies.items():
        try:
            __import__(module)
        except ImportError:
            missing.append(package)
    if missing:
        for pkg in missing:
            subprocess.run([sys.executable, "-m", "pip", "install", pkg, "--break-system-packages"], check=True)

fast_dependency_check()

def get_optimal_config():
    """Velger modell basert på tilgjengelig RAM"""
    total_ram = psutil.virtual_memory().total / (1024**3)
    if total_ram < 3.5: chosen_model = "base"
    elif total_ram < 7.5: chosen_model = "small"
    else: chosen_model = "medium"
    return chosen_model

def get_db_connection():
    """Oppretter forbindelse til MySQL"""
    load_dotenv(ENV_PATH)
    return mysql.connector.connect(
        host=os.getenv("DB_HOST", "localhost"),
        user=os.getenv("DB_USER"),
        password=os.getenv("DB_PASSWORD"),
        database=os.getenv("DB_NAME"),
        charset='utf8mb4',
        collation='utf8mb4_unicode_ci'
    )

def archive_files():
    """Transkriberer VHF-lydfiler med Whisper AI"""
    if not os.path.exists(RECORD_DIR): return
    meta_file = os.path.join(RECORD_DIR, "recordings_meta.log")
    if not os.path.exists(meta_file):
        print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] [-] Ingen metadata-logg funnet.")
        return

    current_hour = datetime.now().hour
    # Tillat manuell kjøring hvis man starter skriptet selv, ellers sjekk time
    if len(sys.argv) < 2 and current_hour != TRANSCRIPTION_HOUR:
        print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] ⏰ AI-plan: kl. {TRANSCRIPTION_HOUR}:00. Hopper over nå.")
        return

    metadata_lookup = {}
    if os.path.exists(meta_file):
        with open(meta_file, "r") as f:
            for line in f:
                parts = line.strip().split(";")
                if len(parts) == 5:
                    metadata_lookup[parts[1]] = {
                        "timestamp": parts[0], "frekvens": parts[2],
                        "navn": parts[3], "varighet": parts[4]
                    }

    # Slett tomme filer (søppel-rydding)
    for f in glob.glob(f"{RECORD_DIR}/*.wav"):
        if os.path.getsize(f) < 10240: # Sletter alt under 10KB
            os.remove(f)

    valid_files = glob.glob(f"{RECORD_DIR}/*.wav")
    if not valid_files: return

    print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] [*] Starter AI-prosessering av {len(valid_files)} filer...")
    model = whisper.load_model(get_optimal_config())
    conn = get_db_connection()
    cursor = conn.cursor()

    for file_path in valid_files:
        filename = os.path.basename(file_path)
        if filename not in metadata_lookup: continue
        
        # Sjekk filstørrelse - sikkerhetsbrems for Pi 5
        file_size_mb = os.path.getsize(file_path) / (1024 * 1024)
        if file_size_mb > JUMP_OVER_MD:
            print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] ⚠️ {filename} er for stor ({file_size_mb:.1f}MB). Hopper over.")
            continue

        meta = metadata_lookup[filename]
        try:
            # Optimalisert for Pi 5: Begrens tråder og lås språk
            result = model.transcribe(file_path, fp16=False)
            tekst = result['text'].strip() or "[Ingen tale funnet]"

            query = "INSERT INTO radio_archive (timestamp, frekvens, navn, tekst, varighet, filnavn) VALUES (%s, %s, %s, %s, %s, %s)"
            cursor.execute(query, (meta["timestamp"], meta["frekvens"], meta["navn"], tekst, meta["varighet"], filename))
            conn.commit()

            print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] ✅ Arkivert: {meta['navn']}")
            if DELETE_AFTER_PROCESSING:
                os.remove(file_path)
        except Exception as e:
            print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] ❌ Feil ved {filename}: {e}")

    open(meta_file, 'w').close() # Tøm metadata
    cursor.close()
    conn.close()

def archive_navtex():
    """Arkiverer NAVTEX-tekstlogger til MySQL"""
    navtex_files = glob.glob(f"{RECORD_DIR}/navtex_*.log")
    if not navtex_files: return

    conn = get_db_connection()
    cursor = conn.cursor()

    for file_path in navtex_files:
        filename = os.path.basename(file_path)
        try:
            with open(file_path, "r") as f:
                innhold = f.read().strip()

            # Rens loggen for teknisk info før lagring
            if not innhold or (len(innhold) < 150 and "Output at 22050 Hz" in innhold):
                os.remove(file_path)
                continue

            # Finn dato fra filnavn
            parts = filename.replace(".log", "").split("_")
            timestamp = datetime.strptime(f"{parts[1]} {parts[2]}", '%Y%m%d %H%M')

            query = "INSERT INTO radio_archive (timestamp, frekvens, navn, tekst, varighet, filnavn) VALUES (%s, %s, %s, %s, %s, %s)"
            cursor.execute(query, (timestamp, "518k", "NAVTEX Rogaland", innhold, 0, filename))
            conn.commit()
            
            print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] ✅ NAVTEX lagret: {filename}")
            if DELETE_AFTER_PROCESSING:
                os.remove(file_path)
        except Exception as e:
            print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] ❌ NAVTEX-feil ved {filename}: {e}")

    cursor.close()
    conn.close()

if __name__ == "__main__":
    # Kjør arkivering. Bruk argumentet 'force' for å tvinge AI-kjøring uansett klokkeslett
    archive_navtex()
    archive_files()
