import subprocess
import time
import sys
import signal
import re
import os
import argparse
import json
import threading
import datetime

# --- AUTOMATISK INSTALLASJON AV GTTS ---
def install_and_import(package):
    try:
        __import__(package)
    except ImportError:
        print(f"[!] Mangler {package}. Installerer nå...")
        try:
            # Vi bruker sys.executable for å sikre at vi bruker samme python-versjon som skriptet
            subprocess.check_call([sys.executable, "-m", "pip", "install", package, "--break-system-packages"])
            print(f"[*] {package} er installert.")
            globals()[package] = __import__(package)
        except Exception as e:
            print(f"[!] Kunne ikke installere {package}: {e}")

# Sjekk og installer gtts
install_and_import('gtts')
from gtts import gTTS

# =================================================================
# --- KONFIGURASJON & DYNAMISK ENHETSHÅNDTERING (v2.0) ---
# =================================================================

def get_usb_device():
    """Leter etter 'UACDemoV10' i aplay -l og returnerer plughw:X,0"""
    try:
        result = subprocess.run(['aplay', '-l'], capture_output=True, text=True)
        for line in result.stdout.splitlines():
            if "UACDemoV10" in line or "USB Audio" in line:
                match = re.search(r'card (\d+):', line)
                if match:
                    card_no = match.group(1)
                    return f"plughw:{card_no},0"
    except: pass
    return "plughw:0,0"

# Denne variabelen brukes nå i alle aplay-kommandoer
AUDIO_DEVICE = get_usb_device()
# Henter ut kortnummeret for bruk i amixer
try:
    CARD_ID = AUDIO_DEVICE.split(':')[1].split(',')[0]
except:
    CARD_ID = "0"

# --- OPPSETT FOR OPPTAK ---
RECORD_DIR = "/home/pi/radio_audio/recordings"
if not os.path.exists(RECORD_DIR):
    os.makedirs(RECORD_DIR)

# --- LYDKONFIGURASJON ---
OUT_RATE = "44100"

# --- LYDSJEKK TIDER ---
TIMES_WEEKDAY = ["1628", "1815", "2015", "2115"]
TIMES_WEEKEND = ["0915", "1115", "1515", "1629", "1815", "2015", "2115"]

# --- TUNER INNSTILLINGER ---
SAMPLE_RATE_COMM = "170k" 
AUDIO_RATE_COMM = "44100"
ENABLE_SOUNDCHECK = True
SNR_BARRIER = 10.0
SQUELCH_TECHNICAL = "25"
SQUELCH_COMMERCIAL = "0"
SCAN_DELAY_SEC = "4"
SEARCH_DURATION = "3s"
FORCE_RESET_SEC = 2
DEFAULT_VOLUME = 70
DEFAULT_GAIN = 30
SAMPLE_RATE_TECH = "24k"
AUDIO_RATE_TECH = "44100"

CACHE_FILE = "last_search.json"

# --- KANALLISTE (VHF + ARBEIDSKANALER v2.2) ---
DEFAULT_CHANNELS = {
    "fm": {
        # --- Eksisterende og korrigerte kanaler ---
        "156.800M": "VHF Nødkanal 16",
        "161.600M": "Arbeidskanal 20 (Tjøme)",
        "161.700M": "Arbeidskanal 22 (Tjøme)",
        "161.675M": "Arbeidskanal 81 (Tjøme)",
        "160.775M": "Arbeidskanal 63 (Tjøme)",
        "156.600M": "Kanal 12 Oslo Havn",
        "156.625M": "Arbeidskanal 72",
        "156.550M": "VHF Kanal 11",
        "156.450M": "VHF Kanal 09",
        "145.600M": "RV44 Tryvann Repeater",
        "145.675M": "Toåsen Nesodden Repeater",
        "145.700M": "RV56 Drammen Repeater",
        "143.900M": "Jakt 1",
        "143.950M": "Jakt 2",
        "444.600M": "KDR Kanal 1",
        "446.006M": "PMR Kanal 1",

        # --- Nye nasjonale arbeidskanaler (fyller hullene) ---
        "156.350M": "VHF Kanal 07 (Arbeid)",
        "161.650M": "VHF Kanal 21 (Arbeid)",
        "161.750M": "VHF Kanal 23 (Arbeid)",
        "161.850M": "VHF Kanal 25 (Arbeid)",
        "161.900M": "VHF Kanal 26 (Arbeid)",
        "160.725M": "VHF Kanal 62 (Arbeid)",
        "160.825M": "VHF Kanal 64 (Arbeid)",
        "160.925M": "VHF Kanal 66 (Arbeid)"
    },
    "am": {
        "118.100M": "Gardermoen Tårn",
        "119.100M": "Oslo Innflyvning",
        "121.500M": "Fly Nødkanal",
        "120.450M": "Oslo Kontroll Sør"
    }
}

COMMERCIAL_CANDIDATES = [
    {"freq": 105.8, "name": "Radio Latin-Amerika"}, {"freq": 106.8, "name": "Radio Metro"},
    {"freq": 101.1, "name": "Radio Riks"}, {"freq": 104.8, "name": "The Beat"},
    {"freq": 99.3,  "name": "Radio Nova"}, {"freq": 107.7, "name": "Norea+"},
    {"freq": 101.6, "name": "P7 Kristen Riks"}, {"freq": 105.3, "name": "Inter FM"},
    {"freq": 91.2,  "name": "Radio L (Asker/Lier)"}, {"freq": 104.5, "name": "pTro"},
    {"freq": 106.1, "name": "Radio Sky"}, {"freq": 91.8,  "name": "Lokalradio"},
    {"freq": 102.0, "name": "Radio Visjon"}, {"freq": 93.3,  "name": "P8 Pop (Lokal)"},
    {"freq": 90.1,  "name": "901 ROX"}
]

# =================================================================

def load_config():
    if os.path.exists("radio_config.json"):
        with open("radio_config.json", "r") as f: return json.load(f)
    return DEFAULT_CHANNELS

def speak(text):
    print(f"-> Tale: {text}")
    try:
        tts = gTTS(text=text, lang='no')
        tts.save("/tmp/v.mp3")
        subprocess.run(f"ffmpeg -y -i /tmp/v.mp3 -ar {OUT_RATE} -ac 2 /tmp/v.wav >/dev/null 2>&1", shell=True)
        subprocess.run(f"aplay -q -D {AUDIO_DEVICE} /tmp/v.wav >/dev/null 2>&1", shell=True)
        time.sleep(0.8) # Pause for ALSA release
    except Exception as e:
        print(f"[!] Tale-feil: {e}")

def force_release_resources():
    print(f"[*] Klargjør maskinvare (bruker {AUDIO_DEVICE})...")
    #subprocess.run("pm2 stop all", shell=True, capture_output=True)
    subprocess.run("sudo killall -9 rtl_fm rtl_power rtl_ais aplay play sox 2>/dev/null", shell=True)
    
    lsmod_check = subprocess.run("lsmod", capture_output=True, text=True)
    if "rtl2832" in lsmod_check.stdout:
        print("[!] Fant blokkerende kjerne-moduler. Frigjør...")
        modules_to_remove = ["rtl2832_sdr", "dvb_usb_rtl28xxu", "rtl2832"]
        for mod in modules_to_remove:
            subprocess.run(f"sudo modprobe -r {mod} 2>/dev/null", shell=True)
    
    time.sleep(1)
    #subprocess.run(f"amixer -c {CARD_ID} sset 'PCM' {DEFAULT_VOLUME}% >/dev/null 2>&1", shell=True)
    time.sleep(FORCE_RESET_SEC)

def restart_services():
    print("\n" + "="*50)
    speak("Går tilbake til normal drift.")
    force_release_resources()
    subprocess.run("pm2 start all", shell=True)

def run_commercial_search():
    print(f"[*] SØK: Analyserer {len(COMMERCIAL_CANDIDATES)} kanaler...")
    speak("Søker etter kommersielle stasjoner.")
    force_release_resources()
    results = []
    
    for ch in COMMERCIAL_CANDIDATES:
        f = ch['freq']
        cmd = f"rtl_power -f {f-0.05}M:{f+0.05}M:2k -i {SEARCH_DURATION} -e {SEARCH_DURATION} -g {DEFAULT_GAIN} /tmp/s.csv"
        subprocess.run(cmd, shell=True)
        time.sleep(0.1)
        
        if os.path.exists("/tmp/s.csv"):
            try:
                with open("/tmp/s.csv", "r") as file:
                    content = file.read().strip()
                    if content:
                        line = content.split(",")
                        if len(line) > 6:
                            vals = [float(x) for x in line[6:] if x.strip() and x != "nan"]
                            snr = round(max(vals) - (sum(sorted(vals)[:5]) / 5), 1)
                            if snr >= SNR_BARRIER:
                                results.append({"freq": f"{f}M", "name": ch['name'], "snr": snr})
            except: pass

    results.sort(key=lambda x: x['snr'], reverse=True)
    if results:
        with open(CACHE_FILE, "w") as f:
            json.dump(results, f)
    return results

def main():
    parser = argparse.ArgumentParser(add_help=False)
    parser.add_argument('--search', action='store_true')
    parser.add_argument('--rescan', action='store_true')
    parser.add_argument('--freq', type=str)
    parser.add_argument('--mode', type=str, default='fm', choices=['fm', 'am'])
    parser.add_argument('--mod', type=str)
    parser.add_argument('--squelch', type=str, default=SQUELCH_TECHNICAL)
    parser.add_argument('--volume', type=int, default=DEFAULT_VOLUME)
    args = parser.parse_args()

    config = load_config()
    is_exiting = [False]

    def safe_exit(s=None, f=None):
        if not is_exiting[0]:
            is_exiting[0] = True
            print("\n[*] Avslutter kontrollert...")
            restart_services()
            sys.exit(0)

    # --- SØKEMODUS ---
    if args.search or args.rescan:
        channels = []
        if not args.rescan and os.path.exists(CACHE_FILE) and os.path.getsize(CACHE_FILE) > 2:
            try:
                print("[*] Laster lagret kanalliste...")
                with open(CACHE_FILE, "r") as f: channels = json.load(f)
                if channels: speak("Laster lagrede stasjoner.")
                else: channels = run_commercial_search()
            except: channels = run_commercial_search()
        else:
            channels = run_commercial_search()
        
        if not channels:
            speak("Ingen stasjoner funnet.")
            print("[!] Ingen stasjoner funnet."); restart_services(); return

        count = len(channels)
        while True:
            print(f"\n{'#':<3} {'FREQ':<10} {'STASJON':<25} {'SNR'}")
            for i, ch in enumerate(channels, 1):
                print(f"{i:<3} {ch['freq']:<10} {ch['name']:25} {ch['snr']}dB")

            try:
                valg = input("\nVelg (#/q/rescan): ").lower()
            except KeyboardInterrupt:
                safe_exit()

            if valg == 'q': break
            if valg == 'rescan':
                channels = run_commercial_search()
                count = len(channels)
                continue

            try:
                idx = int(valg)
                if 1 <= idx <= count:
                    c = channels[idx-1]
                    force_release_resources()
                    print(f"[*] Lytter på {c['name']} ({c['freq']}). Trykk Ctrl+C for meny.")
                    cmd = f"rtl_fm -f {c['freq']} -M wfm -s {SAMPLE_RATE_COMM} -r {OUT_RATE} -g {DEFAULT_GAIN} | aplay -D {AUDIO_DEVICE} -r {OUT_RATE} -f S16_LE -t raw -c 2 -B 500000 2>/dev/null"
                    subprocess.run(cmd, shell=True)
                    speak("Du kan nå velge en ny kanal fra listen.")
            except: continue

        safe_exit()
        return

    # --- VANLIG SKANNING / FREKVENS ---
    signal.signal(signal.SIGINT, safe_exit)

    while True:
        if ENABLE_SOUNDCHECK:
            na = datetime.datetime.now()
            dag = na.weekday()
            klokka = na.strftime("%H%M")
            if (dag < 5 and klokka in TIMES_WEEKDAY) or (dag >= 5 and klokka in TIMES_WEEKEND):
                test_freq = "106.8M"
                speak(f"Planlagt lydsjekk på {test_freq}")
                test_cmd = f"rtl_fm -f {test_freq} -M wfm -s {SAMPLE_RATE_COMM} -r {OUT_RATE} -g {DEFAULT_GAIN} -E 60s | aplay -D {AUDIO_DEVICE} -r {OUT_RATE} -f S16_LE -t raw -c 2 -B 500000 2>/dev/null"
                subprocess.run(test_cmd, shell=True)
                speak("Lydsjekk fullført.")
                time.sleep(2)

        force_release_resources()

        # Opptaksfil for denne økten
        timestamp = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
        rec_file = f"{RECORD_DIR}/scan_{timestamp}.wav"

        is_comm = False
        if args.freq:
            try:
                f_val = float(args.freq.upper().replace('M', ''))
                if 88.0 <= f_val <= 108.0: is_comm = True
            except: pass

        chosen_mod = args.mod if args.mod else ('w' if is_comm else 'n')
        
        if chosen_mod == 'w':
            m_type, s_rate, sq = "wfm", SAMPLE_RATE_COMM, SQUELCH_COMMERCIAL
            pipe_cmd = ""
        else:
            m_type, s_rate, sq = args.mode, SAMPLE_RATE_TECH, args.squelch
            # SOX Resampling + TEE (Opptak) + WAV konvertering for høyttalerstøtte
            pipe_cmd = f" | sox -t raw -r 24k -e signed-integer -b 16 -c 1 - -t wav -r {OUT_RATE} -c 2 - | tee {rec_file}"

        if args.freq:
            f_args = f"-f {args.freq if args.freq.endswith('M') else args.freq+'M'}"
            speak(f"Lytter på {args.freq}")
        else:
            mode_data = config.get(args.mode, {})
            f_list = list(mode_data.keys())
            f_args = " ".join([f"-f {f}" for f in f_list])
            speak(f"Skanner {len(f_list)} kanaler")

        print(f"[*] RADIO AKTIV ({m_type.upper()}). Opptak: {rec_file}")
        
        # Den komplette kommandoen med SoX-piping og TEE-opptak integrert
        full_cmd = f"rtl_fm {f_args} -M {m_type} -s {s_rate} -r {s_rate} -l {sq} -t {SCAN_DELAY_SEC} -g {DEFAULT_GAIN} {pipe_cmd} | aplay -D {AUDIO_DEVICE} -r {OUT_RATE} -f S16_LE 2>/dev/null"

        try:
            subprocess.run(full_cmd, shell=True)
            if args.freq: break
        except KeyboardInterrupt:
            break

    if not is_exiting[0]: safe_exit()

if __name__ == "__main__":
    main()
