import os
import ftplib
import datetime
import subprocess
import sys
from dotenv import load_dotenv

# 1. Konfigurasjon og Lasting av miljøvariabler
load_dotenv('/home/pi/key_info.env')

FTP_HOST = os.getenv("FTP_HOST")
FTP_USER = os.getenv("FTP_USER")
FTP_PASS = os.getenv("FTP_PASSWORD")

# Innstillinger
BASE_FTP_DIR = "radio_audio"
HOME_DIR = "/home/pi"
BACKUP_NAME = "ra_project.tar.gz"
TEMP_ARCHIVE = f"{HOME_DIR}/temp_{BACKUP_NAME}"
FINAL_NAME = BACKUP_NAME

def create_archive():
    """
    Punkt 1: Komprimerer radio_audio mappen (UTEN recordings) 
    OG alle enkeltfiler i /home/pi/.
    """
    print(f"[*] Komprimerer prosjekt og filer i {HOME_DIR}...")
    try:
        # Finn alle enkeltfiler i /home/pi/ (ikke mapper)
        home_files = [f for f in os.listdir(HOME_DIR) if os.path.isfile(os.path.join(HOME_DIR, f))]
        
        # Bygg tar-kommandoen:
        # --exclude='recordings' fjerner mappen inne i radio_audio
        # radio_audio er mappen som skal med
        cmd = ["tar", "-czf", TEMP_ARCHIVE, "--exclude=recordings", "radio_audio"]
        
        # Legg til enkeltfilene fra home-mappen (men ikke selve backup-filen)
        for file in home_files:
            if file == os.path.basename(TEMP_ARCHIVE):
                continue
            cmd.append(file)
            
        # Kjør kommandoen fra /home/pi
        subprocess.run(cmd, cwd=HOME_DIR, check=True)
        return True
    except Exception as e:
        print(f"[!] Feil ved komprimering: {e}")
        return False

def delete_ftp_dir_recursive(ftp, path):
    """Sletter mapper rekursivt på FTP"""
    try:
        ftp.cwd(path)
        items = ftp.nlst()
        for item in items:
            if item in [".", ".."]: continue
            try:
                ftp.delete(item)
            except:
                delete_ftp_dir_recursive(ftp, item)
        ftp.cwd("..")
        ftp.rmd(path)
    except Exception as e:
        print(f"    Kunne ikke slette {path}: {e}")

def cleanup_old_backups(ftp):
    """Punkt 7: Sletter mapper som er 10 uker gamle"""
    print("[*] Sjekker etter 10 uker gamle mapper på FTP...")
    try:
        ftp.cwd(f"/{BASE_FTP_DIR}")
        folders = ftp.nlst()
        limit_date = datetime.datetime.now() - datetime.timedelta(weeks=10)
        
        for folder in folders:
            if folder.startswith("Uke_"):
                try:
                    parts = folder.split('_')
                    week = int(parts[1])
                    year = int(parts[2])
                    # Beregn dato (Mandag i uken)
                    folder_date = datetime.datetime.strptime(f'{year}-W{week}-1', "%Y-W%W-%w")
                    
                    if folder_date < limit_date:
                        print(f"[-] Sletter 10 uker gammel backup: {folder}")
                        delete_ftp_dir_recursive(ftp, folder)
                except:
                    continue
    except:
        pass

def upload_backup():
    if not create_archive(): return

    try:
        now = datetime.datetime.now()
        week_folder = now.strftime("Uke_%V_%Y")
        time_folder = now.strftime("%H%M%S")
        dest_path = f"/{BASE_FTP_DIR}/{week_folder}/{time_folder}"

        print(f"[*] Kobler til {FTP_HOST}...")
        with ftplib.FTP(FTP_HOST, FTP_USER, FTP_PASS) as ftp:
            # Opprett hele stien steg for steg
            current = ""
            for folder in dest_path.split('/'):
                if not folder: continue
                current += "/" + folder
                try:
                    ftp.mkd(current)
                except:
                    pass
            
            ftp.cwd(dest_path)

            # Punkt 4: Sjekk om filen finnes
            target_name = FINAL_NAME
            if target_name in ftp.nlst():
                target_name = f"copy_{now.strftime('%H%M%S')}_{FINAL_NAME}"

            print(f"[*] Laster opp {target_name}...")
            with open(TEMP_ARCHIVE, 'rb') as f:
                ftp.storbinary(f'STOR {target_name}', f)
            
            # Punkt 6: Slett lokal fil etter suksess
            os.remove(TEMP_ARCHIVE)
            print("[*] Suksess. Lokal fil slettet.")

            # Punkt 7: Opprydding
            cleanup_old_backups(ftp)

    except Exception as e:
        print(f"[!] FTP-feil: {e}")

if __name__ == "__main__":
    upload_backup()
