# api_ais_server.py
#Kjører som sudo systemctl start ais_api.service

from flask import Flask, jsonify
import mysql.connector
import json
from datetime import datetime, timezone # Viktig for tidsstempelformatering
from flask_cors import CORS
import traceback # For detaljert feillogging
from dotenv import load_dotenv
import os # <--- LAGT TIL: Nødvendig for å bruke os.getenv()

env_path = '/home/pi/key_info.env'
load_dotenv(dotenv_path=env_path)

# --- START: Database-konfigurasjon (tilpasset fra bruker) ---
DB_HOST = os.getenv("DB_HOST")
DB_USER = os.getenv("DB_USER")
DB_PASSWORD = os.getenv("DB_PASSWORD")
DB_NAME = "totlando_aisdata"
SHIP_TABLE = "ships"          # Tabell for statisk skipinfo
LIVE_AIS_TABLE = "live_ais_data" # Tabell for live AIS-data
# --- SLUTT: Database-konfigurasjon ---

# Importer Enum-typer hvis du vil oversette koder til tekst i API-et senere
# For eksempel: from pyais.messages import NavigationStatus 

# --- Initialiser Flask-applikasjonen og CORS (KUN ÉN GANG) ---
app = Flask(__name__)
CORS(app) # Aktiverer CORS for alle ruter, nyttig for testing fra ulike kilder

# --- Hjelpefunksjoner og Ruter ---

def connect_db():
    """Kobler til databasen og returnerer en tilkoblingsobjekt."""
    try:
        mydb = mysql.connector.connect(
            host=DB_HOST,
            user=DB_USER,
            password=DB_PASSWORD,
            database=DB_NAME,
            collation='utf8mb4_general_ci', # God praksis for tegnsett
            charset='utf8mb4'
        )
        # print("DEBUG [API]: Databasekobling opprettet.") # For feilsøking
        return mydb
    except mysql.connector.Error as err:
        print(f"FEIL [API - connect_db]: Kunne ikke koble til database: {err}")
        traceback.print_exc() # Logg traceback for tilkoblingsfeil også
        return None

@app.route('/api/live_ais_data', methods=['GET'])
def get_live_ais_data():
    mydb = connect_db()
    if not mydb:
        print("FEIL [API - get_live_ais_data]: Klarte ikke å hente data, databasekobling feilet.")
        return jsonify({"error": "Database connection failed", "ships": []}), 500

    mycursor = None 
    try:
        mycursor = mydb.cursor(dictionary=True) 

        # SQL-spørring: Henter live data og kobler med statisk info fra 'ships'-tabellen.
        # Henter kun skip oppdatert de siste 5 minuttene.
        # Bruker UTC_TIMESTAMP() for sammenligning hvis last_update_time er lagret som UTC.
        query = f"""
            SELECT 
                l.mmsi, l.latitude, l.longitude, l.speed_knots, l.course_deg, 
                l.true_heading_deg, l.nav_status_code, l.ais_message_type, 
                l.additional_info, l.last_update_time,
                s.ship_name, 
                s.callsign, 
                s.ship_type AS static_ship_type,
                s.draught,
                s.length,
                s.imo,
                s.image_url 
            FROM {LIVE_AIS_TABLE} l
            LEFT JOIN {SHIP_TABLE} s ON l.mmsi = s.mmsi
            WHERE l.last_update_time >= UTC_TIMESTAMP() - INTERVAL 5 MINUTE 
            ORDER BY l.last_update_time DESC
        """
        
        # print(f"DEBUG [API]: Kjører spørring (de første 250 tegn): {query[:250]}...")
        mycursor.execute(query)
        ships_from_db = mycursor.fetchall()

        processed_ships = []
        for ship_row in ships_from_db:
            processed_ship = {} 

            processed_ship['mmsi'] = str(ship_row.get('mmsi'))
            
            try:
                processed_ship['latitude'] = float(ship_row['latitude'])
                processed_ship['longitude'] = float(ship_row['longitude'])
            except (ValueError, TypeError, KeyError, AttributeError):
                print(f"ADVARSEL [API]: Mangler/ugyldig lat/lon for MMSI {ship_row.get('mmsi')}. Skip ({ship_row.get('ship_name', 'Ukjent Navn')}) hoppes over. Lat: '{ship_row.get('latitude')}', Lon: '{ship_row.get('longitude')}'")
                continue 

            def safe_float(value, default=None):
                if value is None: return default
                try: return float(value)
                except (ValueError, TypeError): return default

            def safe_int(value, default=None):
                if value is None: return default
                try: return int(value)
                except (ValueError, TypeError): return default

            processed_ship['speed_knots'] = safe_float(ship_row.get('speed_knots'))
            processed_ship['course_deg'] = safe_float(ship_row.get('course_deg'))
            processed_ship['true_heading_deg'] = safe_int(ship_row.get('true_heading_deg'))
            processed_ship['nav_status_code'] = safe_int(ship_row.get('nav_status_code'))
            processed_ship['ais_message_type'] = safe_int(ship_row.get('ais_message_type'))
            # processed_ship['rate_of_turn'] = safe_int(ship_row.get('rate_of_turn')) # Hvis du har denne kolonnen

            processed_ship['ship_name'] = ship_row.get('ship_name')
            processed_ship['callsign'] = ship_row.get('callsign')
            processed_ship['static_ship_type'] = ship_row.get('static_ship_type')

            # --- LINJER FOR IMO, DRAUGHT OG IMAGE_URL ---
            processed_ship['draught'] = safe_float(ship_row.get('draught'))
            processed_ship['length'] = safe_float(ship_row.get('length')) 
            processed_ship['imo'] = safe_int(ship_row.get('imo'))
            processed_ship['image_url'] = ship_row.get('image_url') 
            # --- SLUTT PÅ NYE LINJER ---

            additional_info_str = ship_row.get('additional_info')
            if additional_info_str and isinstance(additional_info_str, str):
                try:
                    processed_ship['additional_info'] = json.loads(additional_info_str)
                except json.JSONDecodeError:
                    processed_ship['additional_info'] = {"error_parsing_additional_info": "Kunne ikke parse JSON fra DB"}
            elif additional_info_str: 
                 processed_ship['additional_info'] = additional_info_str
            else:
                processed_ship['additional_info'] = None

            db_timestamp = ship_row.get('last_update_time')
            if db_timestamp and isinstance(db_timestamp, datetime):
                processed_ship['timestamp'] = db_timestamp.replace(tzinfo=timezone.utc).strftime('%Y-%m-%dT%H:%M:%SZ')
            elif isinstance(db_timestamp, str): 
                processed_ship['timestamp'] = db_timestamp 
            else:
                processed_ship['timestamp'] = None
            
            processed_ships.append(processed_ship)
        
        # print(f"DEBUG [API]: Returnerer {len(processed_ships)} skip.")
        return jsonify({"ships": processed_ships})

    except mysql.connector.Error as err:
        print(f"FEIL [API - get_live_ais_data]: Database spørring feilet: {err}")
        traceback.print_exc() 
        return jsonify({"error": "Database query failed", "ships": []}), 500
    except Exception as e:
        print(f"FEIL [API - get_live_ais_data]: Uventet feil: {e} ({type(e).__name__})")
        traceback.print_exc()
        return jsonify({"error": "An unexpected error occurred", "ships": []}), 500
    finally:
        if 'mycursor' in locals() and mycursor: # Sjekk om mycursor ble definert før try-blokken eventuelt feilet
            mycursor.close()
        if 'mydb' in locals() and mydb and mydb.is_connected():
            mydb.close()
            # print("DEBUG [API]: Databasekobling lukket.")

if __name__ == '__main__':
    # For utvikling, debug=True er nyttig. 
    # For mer permanent drift, bruk en produksjons-WSGI-server som Gunicorn.
    # host='0.0.0.0' gjør serveren tilgjengelig på nettverket ditt.
    # Port 5000 er standard for Flask, men kan endres.
    # Husk portvideresending på ruteren din hvis du vil nå den utenfra.
    # Eksempel: Ekstern port 6350 videresendes til intern port 5000 på denne maskinen.
    #app.run(debug=True, host='0.0.0.0', port=5000)
    app.run(debug=False, host='0.0.0.0', port=5000)
