# Filnavn: smart_display_controller.py
# Versjon: 1.0 - Modulær varsling og skjermvisning
import sys
import os
import time
import subprocess
from datetime import datetime
import ftplib
from pathlib import Path

# --- Automatisk Installasjon ---
def install_dependencies():
    """Installerer nødvendige pakker for lyd, skjermstyring og tidssoner."""
    packages = ["pychromecast", "gTTS", "pytz", "python-dotenv", "requests", "playsound==1.2.2"]
    print("--- Mangler nødvendige pakker. Prøver å installere... ---")
    try:
        command = [sys.executable, "-m", "pip", "install"] + packages + ["--break-system-packages"]
        print(f"Kjører kommando: {' '.join(command)}")
        subprocess.check_call(command)
        print("\n--- Pakkene ble installert vellykket. ---")
        return True
    except Exception as e:
        print(f"\nFEIL: Installasjonen feilet: {e}")
        return False

try:
    import pychromecast
    from gtts import gTTS
    import pytz
    from dotenv import load_dotenv
except ImportError as e:
    print(f"Første import feilet: {e}")
    if install_dependencies():
        print("Vennligst kjør skriptet på nytt for å starte programmet.")
    else:
        print("Avslutter på grunn av manglende pakker.")
    sys.exit(1)

# --- Konfigurasjon ---
# OBS: Plasser denne filen i samme mappe som din 'key_info.env'-fil
# eller spesifiser full sti til .env-filen og audio_path.
CONFIG = {
    "google_home_name": "Kjøkken oppe",
    "env_file_path": "/home/pi/key_info.env",
    "temp_audio_path": "/home/pi/ais/audio_logs/",
    "public_url_base": "https://totland.org/totland.org/kjetil/",
    "QUIET_HOUR_START": "22:00",
    "QUIET_HOUR_END": "07:00",
    "TIMEZONE": "Europe/Oslo"
}

# --- Last inn hemmeligheter ---
try:
    load_dotenv(CONFIG['env_file_path'])
    FTP_HOST = os.getenv("FTP_HOST")
    FTP_USER = os.getenv("FTP_USER")
    FTP_PASSWORD = os.getenv("FTP_PASSWORD")
    if not all([FTP_HOST, FTP_USER, FTP_PASSWORD]):
        raise ValueError("FTP-informasjon mangler i .env-filen")
except Exception as e:
    print(f"KRITISK FEIL: Kunne ikke laste konfigurasjon. Feil: {e}")
    sys.exit(1)

# --- Hjelpefunksjoner ---
def log(message):
    print(f"[{datetime.now().strftime('%H:%M:%S')}] {message}")

def is_quiet_time():
    """Sjekker om det er stilleperiode basert på CONFIG."""
    try:
        tz = pytz.timezone(CONFIG["TIMEZONE"])
        now_time = datetime.now(tz).time()
        start_time = datetime.strptime(CONFIG["QUIET_HOUR_START"], '%H:%M').time()
        end_time = datetime.strptime(CONFIG["QUIET_HOUR_END"], '%H:%M').time()
        if start_time > end_time:
            return now_time >= start_time or now_time <= end_time
        else:
            return start_time <= now_time <= end_time
    except Exception as e:
        log(f"ADVARSEL: Kunne ikke sjekke stilleperiode: {e}")
        return False

def get_chromecast(device_name):
    """Finner og returnerer en navngitt Chromecast-enhet."""
    try:
        chromecasts, browser = pychromecast.get_listed_chromecasts(friendly_names=[device_name])
        if chromecasts:
            return chromecasts[0], browser
    except Exception as e:
        log(f"FEIL: Kunne ikke finne Chromecast: {e}")
    return None, None

# ==============================================================================
# === HOVEDFUNKSJON ============================================================
# ==============================================================================
def notify_and_display(message_text, display_url, display_duration_seconds):
    """
    Sender lydvarsel (hvis ikke stille), viser en URL på en skjerm,
    og setter enheten tilbake til normal etter en gitt tid.
    """
    log(f"Starter varsel med melding: '{message_text}'")
    cast, browser = get_chromecast(CONFIG["google_home_name"])
    if not cast:
        if browser: pychromecast.discovery.stop_discovery(browser)
        log("FEIL: Fant ikke enheten. Avbryter.")
        return False

    filename = f"temp_alert_{int(time.time())}.mp3"
    local_filepath = os.path.join(CONFIG['temp_audio_path'], filename)
    public_audio_url = CONFIG['public_url_base'] + filename
    success = False

    try:
        cast.wait()
        play_sound = not is_quiet_time()

        if play_sound:
            log("Genererer lydfil...")
            gTTS(text=message_text, lang='no').save(local_filepath)

            log("Laster opp lydfil via FTP...")
            with ftplib.FTP(FTP_HOST, FTP_USER, FTP_PASSWORD) as ftp:
                with open(local_filepath, 'rb') as file:
                    ftp.storbinary(f'STOR {filename}', file)

            log("Spiller av lydmelding...")
            mc = cast.media_controller
            mc.play_media(public_audio_url, 'audio/mp3')
            mc.block_until_active()
            time.sleep(len(message_text) * 0.15 + 2) # Estimerer tid for avspilling
        else:
            log("Stillemodus er aktiv, hopper over lydmelding.")

        log(f"Viser URL: {display_url} i {display_duration_seconds} sekunder...")
        from pychromecast.controllers.dashcast import DashCastController
        dash = DashCastController()
        cast.register_handler(dash)
        dash.load_url(display_url)
        time.sleep(display_duration_seconds)

        log("Setter enhet tilbake til normal tilstand.")
        cast.quit_app()
        success = True

    except Exception as e:
        log(f"En feil oppstod under varsling/visning: {e}")
        try:
            cast.quit_app() # Prøv å avslutte uansett
        except:
            pass
    finally:
        log("Rydding fullført.")
        if browser:
            pychromecast.discovery.stop_discovery(browser)
        # Rydd opp lydfil fra FTP og lokalt
        try:
            with ftplib.FTP(FTP_HOST, FTP_USER, FTP_PASSWORD) as ftp:
                ftp.delete(filename)
        except Exception:
            pass # Ignorer feil hvis filen ikke finnes
        if os.path.exists(local_filepath):
            os.remove(local_filepath)

    return success

# --- Eksempel på bruk ---
if __name__ == "__main__":
    # Sørg for at mappen for midlertidige lydfiler finnes
    Path(CONFIG['temp_audio_path']).mkdir(parents=True, exist_ok=True)
    
    # Definer variabler for kallet
    melding_til_bruker = "Dette er en test av det nye varslingssystemet."
    url_som_skal_vises = "https://www.yr.no/nb/v%C3%A6rvarsel/daglig-tabell/1-92416/Norge/Agder/Lillesand/Lillesand"
    tid_i_sekunder = 20

    log("--- Starter demonstrasjon av notify_and_display ---")
    
    # Kall funksjonen med de definerte variablene
    notify_and_display(
        message_text=melding_til_bruker,
        display_url=url_som_skal_vises,
        display_duration_seconds=tid_i_sekunder
    )
    
    log("--- Demonstrasjon fullført ---")
