import subprocess
import time
import sys
import signal
import re
import os
import argparse
import json
import threading

# =================================================================
# --- SENTRALE VARIABLER (FREDET - GULLSTANDARD) ---
# =================================================================
SNR_BARRIER = 10.0          
SQUELCH_TECHNICAL = "25"    
SQUELCH_COMMERCIAL = "0"    
SCAN_DELAY_SEC = "4"        
SEARCH_DURATION = "3s"      
FORCE_RESET_SEC = 2         
DEFAULT_VOLUME = 80         
DEFAULT_GAIN = 30           
SAMPLE_RATE_TECH = "24k"    
AUDIO_RATE_TECH = "24000"   

CACHE_FILE = "last_search.json"

DEFAULT_CHANNELS = {
    "fm": {
        "156.800M": "VHF Nødkanal 16", "156.600M": "Kanal 12 Oslo Havn",
        "156.625M": "Arbeidskanal 72", "156.550M": "VHF Kanal 11",
        "156.450M": "VHF Kanal 09", "145.600M": "RV44 Tryvann Repeater",
        "145.675M": "Toåsen Nesodden Repeater", "145.700M": "RV56 Drammen Repeater",
        "143.900M": "Jakt 1", "143.950M": "Jakt 2", "444.600M": "KDR Kanal 1",
        "446.006M": "PMR Kanal 1"
    },
    "am": {
        "118.100M": "Gardermoen Tårn", "119.100M": "Oslo Innflyvning",
        "121.500M": "Fly Nødkanal", "120.450M": "Oslo Kontroll Sør"
    }
}

COMMERCIAL_CANDIDATES = [
    {"freq": 105.8, "name": "Radio Latin-Amerika"}, {"freq": 106.8, "name": "Radio Metro"},
    {"freq": 101.1, "name": "Radio Riks"}, {"freq": 104.8, "name": "The Beat"},
    {"freq": 99.3,  "name": "Radio Nova"}, {"freq": 107.7, "name": "Norea+"},
    {"freq": 101.6, "name": "P7 Kristen Riks"}, {"freq": 105.3, "name": "Inter FM"},
    {"freq": 91.2,  "name": "Radio L (Asker/Lier)"}, {"freq": 104.5, "name": "pTro"},
    {"freq": 106.1, "name": "Radio Sky"}, {"freq": 91.8,  "name": "Lokalradio"},
    {"freq": 102.0, "name": "Radio Visjon"}, {"freq": 93.3,  "name": "P8 Pop (Lokal)"},
    {"freq": 90.1,  "name": "901 ROX"}
]

# =================================================================

def load_config():
    if os.path.exists("radio_config.json"):
        with open("radio_config.json", "r") as f: return json.load(f)
    return DEFAULT_CHANNELS

def speak(text):
    print(f"-> Tale: {text}")
    def run_speech():
        try:
            from gtts import gTTS
            tts = gTTS(text=text, lang='no')
            tts.save("/tmp/v.mp3")
            subprocess.run(f"ffmpeg -y -i /tmp/v.mp3 -ar {AUDIO_RATE_TECH} -ac 1 /tmp/v.wav >/dev/null 2>&1", shell=True)
            subprocess.run("aplay -q /tmp/v.wav >/dev/null 2>&1", shell=True)
        except: pass
    threading.Thread(target=run_speech).start()

def force_release_resources():
    print(f"[*] Klargjør maskinvare...")
    subprocess.run("pm2 stop all", shell=True, capture_output=True)
    subprocess.run("sudo killall -9 rtl_fm rtl_power rtl_ais aplay 2>/dev/null", shell=True)
    subprocess.run(f"amixer sset 'Master' {DEFAULT_VOLUME}% >/dev/null 2>&1", shell=True)
    time.sleep(FORCE_RESET_SEC)

def restart_services():
    print("\n" + "="*50)
    speak("Går tilbake til normal drift.")
    force_release_resources()
    subprocess.run("pm2 start all", shell=True)

def run_commercial_search():
    print(f"[*] SØK: Analyserer {len(COMMERCIAL_CANDIDATES)} kanaler...")
    speak("Søker etter kommersielle stasjoner.")
    force_release_resources()
    results = []
    for ch in COMMERCIAL_CANDIDATES:
        f = ch['freq']
        cmd = f"rtl_power -f {f-0.1}M:{f+0.1}M:10k -i {SEARCH_DURATION} -e {SEARCH_DURATION} -g {DEFAULT_GAIN} -p 0 /tmp/s.csv"
        subprocess.run(cmd, shell=True)
        time.sleep(0.5)
        if os.path.exists("/tmp/s.csv"):
            try:
                with open("/tmp/s.csv", "r") as file:
                    line = file.read().split(",")
                    if len(line) > 6:
                        vals = [float(x) for x in line[6:] if x.strip() and x != "nan"]
                        snr = round(max(vals) - (sum(sorted(vals)[:5]) / 5), 1)
                        if snr >= SNR_BARRIER:
                            results.append({"freq": f"{f}M", "name": ch['name'], "snr": snr})
            except: pass
    
    results.sort(key=lambda x: x['snr'], reverse=True)
    if results:
        with open(CACHE_FILE, "w") as f:
            json.dump(results, f)
    return results

def main():
    parser = argparse.ArgumentParser(add_help=False)
    parser.add_argument('--search', action='store_true')
    parser.add_argument('--rescan', action='store_true')
    parser.add_argument('--freq', type=str)
    parser.add_argument('--mode', type=str, default='fm', choices=['fm', 'am'])
    parser.add_argument('--mod', type=str)
    parser.add_argument('--squelch', type=str, default=SQUELCH_TECHNICAL)
    parser.add_argument('--volume', type=int, default=DEFAULT_VOLUME)
    args = parser.parse_args()

    config = load_config()
    is_exiting = [False]

    def safe_exit(s=None, f=None):
        if not is_exiting[0]:
            is_exiting[0] = True
            print("\n[*] Avslutter kontrollert...")
            restart_services()
            sys.exit(0)

    # --- SØKEMODUS ---
    if args.search or args.rescan:
        channels = []
        if not args.rescan and os.path.exists(CACHE_FILE) and os.path.getsize(CACHE_FILE) > 2:
            try:
                print("[*] Laster lagret kanalliste...")
                with open(CACHE_FILE, "r") as f: channels = json.load(f)
                if channels: speak("Laster lagrede stasjoner.")
                else: channels = run_commercial_search()
            except: channels = run_commercial_search()
        else:
            channels = run_commercial_search()

        if not channels:
            print("[!] Ingen stasjoner funnet."); restart_services(); return
        
        count = len(channels)
        while True:
            # VIKTIG: Ingen signal-handler her, la subprocess håndtere Ctrl+C
            print(f"\n{'#':<3} {'FREQ':<10} {'STASJON':<25} {'SNR'}")
            for i, ch in enumerate(channels, 1):
                print(f"{i:<3} {ch['freq']:<10} {ch['name']:25} {ch['snr']}dB")
            
            try:
                valg = input("\nVelg (#/q/rescan): ").lower()
            except KeyboardInterrupt:
                # Hvis bruker trykker Ctrl+C i selve input-feltet, avslutt kontrollert
                safe_exit()
            
            if valg == 'q': break
            if valg == 'rescan':
                channels = run_commercial_search()
                count = len(channels)
                continue
                
            try:
                idx = int(valg)
                if 1 <= idx <= count:
                    c = channels[idx-1]
                    force_release_resources()
                    print(f"[*] Lytter på {c['name']} ({c['freq']}). Trykk Ctrl+C for meny.")
                    
                    # Denne vil nå fange Ctrl+C, stoppe radioen, og fortsette loopen
                    cmd = f"rtl_fm -f {c['freq']} -M wfm -s 170k -r 32000 -g {DEFAULT_GAIN} | aplay -r 32000 -f S16_LE -t raw -c 1 -B 500000 2>/dev/null"
                    subprocess.run(cmd, shell=True)
                    
                    speak("Du kan nå velge en ny kanal fra listen, avslutte med kju, eller skanne på nytt med reskan.")
            except: continue
        
        safe_exit()
        return

    # --- VANLIG SKANNING ---
    # Her setter vi på signal-håndteringen igjen for vanlig drift
    signal.signal(signal.SIGINT, safe_exit)
    force_release_resources()
    
    is_comm = False
    if args.freq:
        try:
            f_val = float(args.freq.upper().replace('M', ''))
            if 88.0 <= f_val <= 108.0: is_comm = True
        except: pass

    chosen_mod = args.mod if args.mod else ('w' if is_comm else 'n')
    current_sq = args.squelch
    if chosen_mod == 'w':
        m_type, s_rate, a_rate = "wfm", "170k", "32000"
    else:
        m_type, s_rate, a_rate = args.mode, SAMPLE_RATE_TECH, AUDIO_RATE_TECH

    mode_data = config.get(args.mode, {})
    if args.freq:
        f_args = f"-f {args.freq if args.freq.endswith('M') else args.freq+'M'}"
        speak(f"Lytter på {args.freq}")
    else:
        f_list = list(mode_data.keys())
        f_args = " ".join([f"-f {f}" for f in f_list])
        speak(f"Skanner {len(f_list)} kanaler")

    print(f"[*] RADIO AKTIV ({m_type.upper()}) - Squelch: {current_sq} - Gain: {DEFAULT_GAIN}")
    full_cmd = f"rtl_fm {f_args} -M {m_type} -s {s_rate} -r {a_rate} -l {current_sq} -t {SCAN_DELAY_SEC} -g {DEFAULT_GAIN} | aplay -r {a_rate} -f S16_LE -t raw -c 1 -B 500000 2>/dev/null"
    
    try:
        subprocess.run(full_cmd, shell=True)
    except KeyboardInterrupt: pass
    finally:
        if not is_exiting[0]: safe_exit()

if __name__ == "__main__":
    main()