# Filnavn: ship_alert.py
# Versjon: 52.0 (Forenklet direktevarsling uten ETA)
import sys
import requests
import time
import logging
from datetime import datetime
from haversine import haversine, Unit

# --- Konfigurasjon ---
SHIP_ALERT_CONFIG = {
    "api_url": "http://127.0.0.1:5000/api/live_ais_data",
    "website_to_show": "http://89.10.213.234:6380/",
    "log_file_path": "/home/pi/ais/passage.log",
    "min_draught_alert": 2.0,
    "check_interval_seconds": 15,
    "alert_zone": [
        [59.83882095853034, 10.5099888449554], 
        [59.77202830810234, 10.59253523611673], 
        [59.80641392017936, 10.60936045556268], 
        [59.83882095853034, 10.5099888449554]
    ],
    "valid_course_ranges": { "inbound": (330, 30), "outbound": (150, 210) }
}

# --- Oppsett av varsling (Smart Display) ---
try:
    from smart_display_controller import notify_and_display
except ImportError:
    print("FEIL: Mangler 'smart_display_controller.py'")
    sys.exit(1)

def is_point_in_polygon(point, polygon):
    lat, lon = point; n = len(polygon); inside = False
    p1lat, p1lon = polygon[0]
    for i in range(n + 1):
        p2lat, p2lon = polygon[i % n]
        if min(p1lat, p2lat) < lat <= max(p1lat, p2lat) and lon <= max(p1lon, p2lon):
            if p1lat != p2lat:
                lon_intersection = (lat - p1lat) * (p2lon - p1lon) / (p2lat - p1lat) + p1lon
                if p1lon == p2lon or lon <= lon_intersection:
                    inside = not inside
        p1lat, p1lon = p2lat, p2lon
    return inside

def run_alert_loop():
    notified_ships = {} # MMSI: timestamp
    print(f"[{datetime.now()}] Varslingssystem v52.0 aktivt...")

    while True:
        try:
            response = requests.get(SHIP_ALERT_CONFIG["api_url"], timeout=10)
            if response.status_code == 200:
                ships = response.json().get("ships", [])
                
                for ship in ships:
                    mmsi = ship.get('mmsi')
                    lat, lon = ship.get('latitude'), ship.get('longitude')
                    draught = ship.get('draught') or 0.0
                    
                    # Fallback for navn (Nytt i v52.0)
                    ship_name = ship.get('ship_name') or f"MMSI {mmsi}"

                    if not mmsi or lat is None or lon is None: continue
                    
                    # Sjekk om skipet er i sonen og oppfyller krav
                    if is_point_in_polygon((lat, lon), SHIP_ALERT_CONFIG["alert_zone"]):
                        if draught >= SHIP_ALERT_CONFIG["min_draught_alert"]:
                            
                            # Kun varsle én gang per time per skip
                            if mmsi not in notified_ships or (time.time() - notified_ships[mmsi] > 3600):
                                message = f"ALARM: {ship_name} er nå i indre sone (Dybde: {draught}m)."
                                print(f"[{datetime.now()}] {message}")
                                
                                target_url = f'{SHIP_ALERT_CONFIG["website_to_show"]}index_mmsi.html?mmsi={mmsi}'
                                notify_and_display(message_text=message, display_url=target_url)
                                
                                notified_ships[mmsi] = time.time()
                                
        except Exception as e:
            print(f"Feil i løkke: {e}")
            
        time.sleep(SHIP_ALERT_CONFIG["check_interval_seconds"])

if __name__ == "__main__":
    run_alert_loop()
