# ship_data_updater.py (Din versjon v21 + KUN timeout-fiks)
import time
import mysql.connector
import traceback
import os
import re
from decimal import Decimal
from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from dotenv import load_dotenv
from selenium.webdriver.chrome.service import Service


# --- KONFIGURASJON ---
env_path = '/home/pi/key_info.env'
load_dotenv(dotenv_path=env_path)

DB_HOST = os.getenv("DB_HOST")
DB_USER = os.getenv("DB_USER")
DB_PASSWORD = os.getenv("DB_PASSWORD")
DB_NAME = "totlando_aisdata"
SHIP_TABLE = "ships"
PERIOD = "WEEK"
NO_OF_PERIOD = "1"

def connect_db():
    try:
        return mysql.connector.connect(
            host=DB_HOST, user=DB_USER, password=DB_PASSWORD, database=DB_NAME
        )
    except mysql.connector.Error as err:
        print(f"❌ DB Connect Error: {err}")
        return None

def get_ship_to_update(db_conn):
    try:
        with db_conn.cursor(dictionary=True) as cursor:
            query = f"""
                SELECT mmsi, image_url, draught, length, beam
                FROM {SHIP_TABLE}
                WHERE 
                    (last_updated IS NULL OR last_updated < NOW() - INTERVAL {NO_OF_PERIOD} {PERIOD})
                    AND 
                    (
                        (image_url IS NULL OR (image_url NOT LIKE 'https%%' AND image_url != 'NOT_FOUND1'))
                        OR draught IS NULL OR draught = 0.00
                        OR length IS NULL
                        OR beam IS NULL
                    )
                ORDER BY RAND()
                LIMIT 1;
            """
            cursor.execute(query)
            result = cursor.fetchone()
            return result if result else None
    except mysql.connector.Error as err:
        print(f"❌ DB Query Error (get_ship): {err}")
        return None

def scrape_ship_data(mmsi):
    print(f"📊 Søker etter data for MMSI {mmsi}...")
    
    chrome_options = Options()
    chrome_options.add_argument("--headless")
    chrome_options.add_argument("--no-sandbox")
    chrome_options.add_argument("--disable-dev-shm-usage")
    
    driver = None
    data = {'image_url': None, 'draught': None, 'length': None, 'beam': None}

    service = Service(executable_path='/usr/bin/chromedriver')

    
    # Endringen er kun i try-blokken under
    try:
        driver = webdriver.Chrome(service=service, options=chrome_options)
        url = f"https://nais.kystverket.no/aisstream/{mmsi}"
        driver.get(url)
        wait = WebDriverWait(driver, 25)

        # ====================== DEN ENE ENDRINGEN ER HER ======================
        # Vi legger til en try/except KUN rundt 'wait.until' for å fange timeout
        try:
            print("  - Venter på at siden skal laste innhold...")
            wait.until(EC.presence_of_element_located((By.CSS_SELECTOR, "img.vessel-photo")))
            print("  - Innhold lastet. Henter rå HTML-kode...")
        except Exception:
            # Hvis den timer ut, logg en advarsel, men IKKE krasj. Hopp ut av 'try'-blokken.
            print(f"  ❌ Timeout under lasting av siden for MMSI {mmsi}. Går videre.")
            # Returnerer tom data og går til 'finally' for å lukke nettleseren
            return data
        # ====================================================================

        # Resten av koden din er helt uendret
        html_source = driver.page_source
        
        img_match = re.search(r'<img.*?class="vessel-photo".*?src="(.*?)"', html_source)
        if img_match:
            data['image_url'] = img_match.group(1)
            print(f"  ✅ Fant Bilde-URL: {data['image_url']}")
        else:
            print(f"  ℹ️ Fant ikke bilde-URL i HTML-koden.")

        def get_value_from_html(key_text, html):
            pattern = re.compile(f'<span data-v-.*?">{key_text}</span>.*?<div data-v-.*? class="description">(.*?)</div>', re.DOTALL)
            match = pattern.search(html)
            if match:
                return match.group(1).strip()
            return None
        
        length_text = get_value_from_html('Lengde', html_source)
        if length_text:
            match = re.search(r'(\d+)', length_text)
            if match:
                data['length'] = int(match.group(1))
                print(f"  ✅ Fant Lengde: {data['length']} m")
        else:
            print(f"  ℹ️ Fant ikke Lengde i HTML-koden.")

        beam_text = get_value_from_html('Bredde', html_source)
        if beam_text:
            match = re.search(r'(\d+)', beam_text)
            if match:
                data['beam'] = int(match.group(1))
                print(f"  ✅ Fant Bredde: {data['beam']} m")
        else:
            print(f"  ℹ️ Fant ikke Bredde i HTML-koden.")
            
        draught_text = get_value_from_html('Dypgang', html_source)
        if draught_text:
            match = re.search(r'(\d+\.?\d*)', draught_text)
            if match:
                data['draught'] = Decimal(match.group(1))
                print(f"  ✅ Fant Dypgang: {data['draught']} m")
        else:
            print(f"  ℹ️ Fant ikke Dypgang i HTML-koden.")
            
        return data

    # Din opprinnelige 'except' er fjernet siden vi nå håndterer timeout spesifikt
    # og lar andre feil (som er usannsynlige) stoppe programmet for feilsøking.

    finally:
        if driver:
            driver.quit()

def update_ship_data_in_db(db_conn, mmsi, existing_data, new_data):
    # Denne funksjonen er helt uendret
    update_fields = ["last_updated = CURRENT_TIMESTAMP"]
    update_values = []
    
    new_image_url = new_data.get('image_url') or 'NOT_FOUND1'
    if new_image_url != existing_data.get('image_url'):
        update_fields.append("image_url = %s")
        update_values.append(new_image_url)

    new_draught = new_data.get('draught')
    if new_draught is not None and new_draught != existing_data.get('draught'):
        update_fields.append("draught = %s")
        update_values.append(new_draught)
        
    new_length = new_data.get('length')
    if new_length is not None and new_length != existing_data.get('length'):
        update_fields.append("length = %s")
        update_values.append(new_length)
        
    new_beam = new_data.get('beam')
    if new_beam is not None and new_beam != existing_data.get('beam'):
        update_fields.append("beam = %s")
        update_values.append(new_beam)

    try:
        with db_conn.cursor() as cursor:
            if len(update_fields) > 1:
                 print(f"  ✏️ Funnet endringer for MMSI {mmsi}. Oppdaterer databasen...")
            else:
                 print(f"  👍 Ingen nye datafunn for MMSI {mmsi}. Oppdaterer kun 'last_updated'.")

            query = f"UPDATE {SHIP_TABLE} SET {', '.join(update_fields)} WHERE mmsi = %s;"
            update_values.append(mmsi)
            
            cursor.execute(query, tuple(update_values))
            db_conn.commit()
            print(f"  💾 Databasen er oppdatert for MMSI {mmsi}. (Sist sjekket: nå)")
            
    except mysql.connector.Error as err:
        print(f"❌ DB Update Error: {err}")

def main():
    # Denne funksjonen er helt uendret
    print("--- Starter Skipsdata-oppdaterer (v21: Forenklet) ---")
    while True:
        db_conn = connect_db()
        if not db_conn:
            time.sleep(60)
            continue
            
        ship_to_process = get_ship_to_update(db_conn)
        
        if ship_to_process:
            mmsi = ship_to_process['mmsi']
            new_ship_data = scrape_ship_data(mmsi)
            update_ship_data_in_db(db_conn, mmsi, ship_to_process, new_ship_data)
        else:
            print("✅ Ingen skip trenger oppdatering for øyeblikket. Avslutter.")
            if db_conn.is_connected(): db_conn.close()
            break
        
        if db_conn.is_connected():
            db_conn.close()
    
        pause_duration = 25
        print(f"--- Venter i {pause_duration} sekunder... --- \n")
        time.sleep(pause_duration)

if __name__ == "__main__":
    # Denne blokken er helt uendret
    try:
        main()
    except KeyboardInterrupt:
        print("\nAvslutter manuelt.")
    except Exception as e:
        print(f"💥 En kritisk feil oppstod: {e}")
        traceback.print_exc()