import ftplib
import os
import tarfile
import glob
import re
from datetime import datetime, timedelta
from dotenv import load_dotenv

# --- KONFIGURASJON ---
load_dotenv('/home/pi/key_info.env')

FTP_HOST = "totland.org"
FTP_USER = os.getenv("FTP_USER")
FTP_PASS = os.getenv("FTP_PASSWORD")

# Ukesmappe: rsp_backup_W02_2026
WEEK_FOLDER = datetime.now().strftime("rsp_backup_W%V_%Y")
BASE_DIR = "radio_backup"
ARCHIVE_NAME = "ais_project.tar.gz"

# Områder som skal inkluderes
AIS_DIR = "/home/pi/ais"
WEB_DIR = "/var/www/html"
HOME_FILES = "/home/pi/*.*"

def delete_old_backups(session):
    """Sletter mapper som er eldre enn 10 uker"""
    print("--- Sjekker etter gamle backuper (10 uker) ---")
    try:
        session.cwd(f"/{BASE_DIR}")
        directories = session.nlst()
        
        ten_weeks_ago = datetime.now() - timedelta(weeks=10)
        
        for folder in directories:
            # Matcher mønsteret rsp_backup_W[uke]_[år]
            match = re.match(r"rsp_backup_W(\d{2})_(\d{4})", folder)
            if match:
                week_num = int(match.group(1))
                year_num = int(match.group(2))
                
                # Lager en omtrentlig dato for mappen (mandag i den uken)
                folder_date = datetime.strptime(f'{year_num}-W{week_num}-1', "%Y-W%W-%w")
                
                if folder_date < ten_weeks_ago:
                    print(f"🗑️ Sletter gammel mappe: {folder}")
                    try:
                        # Sletter filer i mappen før mappen slettes
                        session.cwd(folder)
                        for f in session.nlst():
                            session.delete(f)
                        session.cwd("..")
                        session.rmd(folder)
                    except Exception as e:
                        print(f"Kunne ikke slette {folder}: {e}")
    except Exception as e:
        print(f"Feil under sletting av gamle mapper: {e}")

def create_archive():
    try:
        print(f"--- Pakker {ARCHIVE_NAME} ---")
        with tarfile.open(ARCHIVE_NAME, "w:gz") as tar:
            if os.path.exists(AIS_DIR):
                tar.add(AIS_DIR, arcname="ais")
            if os.path.exists(WEB_DIR):
                tar.add(WEB_DIR, arcname="html")
            for file in glob.glob(HOME_FILES):
                if os.path.isfile(file):
                    tar.add(file, arcname=os.path.basename(file))
        return True
    except Exception as e:
        print(f"❌ Feil under pakking: {e}")
        return False

def upload_to_totland():
    if not create_archive(): return

    try:
        print(f"🚀 Kobler til {FTP_HOST}...")
        session = ftplib.FTP(FTP_HOST, FTP_USER, FTP_PASS)
        
        # 1. Rydd opp gamle backuper først
        delete_old_backups(session)
        
        # 2. Gå til ukes-mappen for opplasting
        session.cwd(f"/{BASE_DIR}")
        try:
            session.cwd(WEEK_FOLDER)
        except:
            print(f"Oppretter ny ukesmappe: {WEEK_FOLDER}")
            session.mkd(WEEK_FOLDER)
            session.cwd(WEEK_FOLDER)

        print(f"⬆️ Overfører {ARCHIVE_NAME}...")
        with open(ARCHIVE_NAME, "rb") as file:
            session.storbinary(f"STOR {ARCHIVE_NAME}", file)
        
        session.quit()
        print(f"✅ Fullført! Filen ligger i: {BASE_DIR}/{WEEK_FOLDER}")
        
        if os.path.exists(ARCHIVE_NAME):
            os.remove(ARCHIVE_NAME)
    except Exception as e:
        print(f"❌ FTP-feil: {e}")

if __name__ == "__main__":
    upload_to_totland()
