import os
import ftplib
import datetime
import subprocess
from pathlib import Path
from dotenv import load_dotenv

# Laster .env-filen fra en fast, absolutt sti.
try:
    env_path = '/home/pi/key_info.env'
    load_dotenv(dotenv_path=env_path)
    print(f"Laster inn konfigurasjon fra: {env_path}")
except Exception as e:
    print(f"ADVARSEL: Kunne ikke laste .env-fil. Feil: {e}")

# --- Konfigurasjon ---
FTP_HOST = os.getenv("FTP_HOST")
FTP_USER = os.getenv("FTP_USER")
FTP_PASSWORD = os.getenv("FTP_PASSWORD")

LOCAL_TAR_PATH = '/home/pi/ais_project.tar.gz'
REMOTE_BASE_DIR = '/radio_backup/'
FOLDER_PREFIX = 'rsp_backup_' 
WEEKS_TO_KEEP = 10

def delete_folder_recursively(ftp, path):
    """Sletter en mappe og alt dens innhold rekursivt på en FTP-server."""
    try:
        ftp.cwd(path)
    except ftplib.error_perm:
        return

    items = ftp.nlst()
    items_to_process = [item for item in items if item not in ('.', '..', path, os.path.basename(path))]
    
    for item in items_to_process:
        full_item_path = os.path.join(path, os.path.basename(item))
        try:
            ftp.cwd(item)
            ftp.cwd('..')
            delete_folder_recursively(ftp, full_item_path)
        except ftplib.error_perm:
            ftp.delete(item) 

    try:
        ftp.cwd('..')
        ftp.rmd(path)
        print(f"Slettet gammel backup-mappe: {path}")
    except Exception as e:
        print(f"Kunne ikke slette mappen {path}: {e}")

def upload_backup_file(ftp, local_file, remote_dir):
    """Laster opp tar.gz filen til spesifisert mappe."""
    if not os.path.isfile(local_file):
        print(f"FEIL: Backup-filen {local_file} ble ikke funnet.")
        return False
        
    filename = os.path.basename(local_file)
    
    try:
        # Opprett fjernmappe om den ikke finnes
        current_dir = ""
        for part in remote_dir.split('/'):
            if not part: continue
            current_dir += f"/{part}"
            try:
                ftp.cwd(current_dir)
            except ftplib.error_perm:
                ftp.mkd(current_dir)
                ftp.cwd(current_dir)

        with open(local_file, 'rb') as f:
            print(f"Laster opp backup-arkiv: {filename}...")
            ftp.storbinary(f'STOR {filename}', f)
            return True
    except Exception as e:
        print(f"FEIL under opplasting: {e}")
        return False

def main():
    # 1. Komprimer prosjektet
    print("\n--- Steg 1: Komprimerer prosjektet ---")
    tar_cmd = (
        f"sudo tar -czvf {LOCAL_TAR_PATH} "
        f"--exclude='{LOCAL_TAR_PATH}' "
        f"/home/pi/ais /var/www/html /home/pi/*.*"
    )
    
    try:
        subprocess.run(tar_cmd, shell=True, check=True)
        print("Komprimering fullført.")
    except subprocess.CalledProcessError as e:
        print(f"Kritisk feil under komprimering: {e}")
        return

    # 2. FTP-håndtering
    today = datetime.date.today()
    current_year, current_week, _ = today.isocalendar()
    date_to_delete = today - datetime.timedelta(weeks=WEEKS_TO_KEEP)
    delete_year, delete_week, _ = date_to_delete.isocalendar()

    new_folder = f"{FOLDER_PREFIX}W{current_week:02d}_{current_year}"
    old_folder = f"{FOLDER_PREFIX}W{delete_week:02d}_{delete_year}"
    
    remote_dest_path = os.path.join(REMOTE_BASE_DIR, new_folder)
    remote_old_path = os.path.join(REMOTE_BASE_DIR, old_folder)

    if not all([FTP_HOST, FTP_USER, FTP_PASSWORD]):
        print("Feil: Mangler FTP-konfigurasjon i .env")
        return

    try:
        with ftplib.FTP(FTP_HOST, FTP_USER, FTP_PASSWORD) as ftp:
            print(f"Koblet til {FTP_HOST}")
            
            # Last opp filen
            success = upload_backup_file(ftp, LOCAL_TAR_PATH, remote_dest_path)
            
            if success:
                print("Opplasting vellykket!")
                # 3. Slett gammel backup kun hvis den nye ble lastet opp OK
                print(f"Sjekker etter gammel backup: {old_folder}")
                delete_folder_recursively(ftp, remote_old_path)
            else:
                print("Opplasting feilet. Beholder gammel backup for sikkerhets skyld.")

    except Exception as e:
        print(f"FTP-feil: {e}")

if __name__ == '__main__':
    main()
