import subprocess
import socket
import json
import os
import signal
import psutil
import time
import requests
import traceback
import mysql.connector

from datetime import datetime, timedelta, timezone
from pyais import decode
from pyais.messages import NavigationStatus, ManeuverIndicator, EpfdType, NavAid, TurnRate
# from datetime import datetime, timedelta, timezone # Allerede importert
from pyais.exceptions import MissingMultipartMessageException

from dotenv import load_dotenv
env_path = '/home/pi/key_info.env'
load_dotenv(dotenv_path=env_path)


print("--- SCRIPTET HAR STARTET ---")

# ===================================================================
# KJØRES MED PM2 (Process Manager 2)
# -------------------------------------------------------------------
# Start:      pm2 start ais_to_json.py --interpreter python3 --name ais-collector
# Se status:  pm2 list
# Se logg:    pm2 logs ais-collector --lines 100
# Restart:    pm2 restart ais-collector
# Stopp:      pm2 stop ais-collector
# ===================================================================

# KONFIGURASJON: Timeout for skip i JSON (minutter)
SHIP_TIMEOUT_MINUTES = 30 # Skip fjernes fra JSON hvis siste posisjon er eldre enn dette


# KONFIGURASJON: AISHub API-nøkkel (fyll inn for å aktivere API - MERK: Ikke brukt for UDP i denne koden)
AISHUB_API_KEY = "aktiver_aiahub_udp"  # La stå tom hvis ikke relevant
SHIP_IMAGE_FOLDER = "/var/www/html/shipimage"

# Filbaner
JSON_FILE = "/var/www/html/ais_data.json"
UDP_PORT = 10110  # Lokal port for mottak fra rtl_ais

# KONFIGURASJON: Database-detaljer
DB_HOST = os.getenv("DB_HOST")
DB_USER = os.getenv("DB_USER")
DB_PASSWORD = os.getenv("DB_PASSWORD")
DB_NAME = "totlando_aisdata"
SHIP_TABLE = "ships"

# KONFIGURASJON: AISHub UDP-detaljer
AISHUB_IP = "144.76.105.244" # Standard AISHub IP
AISHUB_PORT = 3505          # Standard AISHub port

def is_running_as_systemd_service():
    """
    Sjekker om skriptet sannsynligvis kjører som en systemd-tjeneste
    ved å se etter INVOCATION_ID miljøvariabelen.
    """
    return "INVOCATION_ID" in os.environ

def connect_db():
    """Oppretter en tilkobling til MariaDB-databasen."""
    try:
        mydb = mysql.connector.connect(
            host=DB_HOST,
            user=DB_USER,
            password=DB_PASSWORD,
            database=DB_NAME,
        )
        return mydb
    except mysql.connector.Error as err:
        print(f"❌ Feil ved tilkobling til databasen: {err}")
        return None

def create_ship_table():
    """Oppretter tabellen 'ships' hvis den ikke allerede eksisterer,
       og sikrer at den har kolonnen 'ship_type_code'."""
    mydb = connect_db()
    if mydb is None:
        return
    mycursor = mydb.cursor()
    try:
        # 1. Opprett tabellen med den nye kolonnen hvis den ikke finnes
        mycursor.execute(
            f"""
            CREATE TABLE IF NOT EXISTS {SHIP_TABLE} (
                mmsi VARCHAR(15) PRIMARY KEY,
                callsign VARCHAR(10),
                ship_name VARCHAR(255),
                ship_type_code INT NULL,         -- NY KOLONNE for numerisk skipstype
                ship_type VARCHAR(255),          -- Eksisterende kolonne for tekstbeskrivelse
                image_url VARCHAR(1024),
                last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )
            """
        )
        mydb.commit()
        print(f"✅ Tabell '{SHIP_TABLE}' sjekket/opprettet initielt.")

        # 2. Sjekk om kolonnen 'ship_type_code' eksisterer hvis tabellen allerede fantes
        mycursor.execute(f"SHOW COLUMNS FROM `{SHIP_TABLE}` LIKE 'ship_type_code';")
        column_exists = mycursor.fetchone()

        if not column_exists:
            print(f"⚠️ Kolonnen 'ship_type_code' mangler i '{SHIP_TABLE}'. Legger den til...")
            # Plasserer den logisk før den tekstlige ship_type for lesbarhet i databasen
            mycursor.execute(f"ALTER TABLE `{SHIP_TABLE}` ADD COLUMN ship_type_code INT NULL AFTER ship_name;")
            mydb.commit()
            print(f"✅ Kolonnen 'ship_type_code' lagt til i tabellen '{SHIP_TABLE}'.")
        else:
            print(f"✅ Kolonnen 'ship_type_code' eksisterer allerede i '{SHIP_TABLE}'.")

    except mysql.connector.Error as err:
        print(f"❌ Feil ved klargjøring av tabell '{SHIP_TABLE}': {err}")
        traceback.print_exc()
    finally:
        if mycursor:
            mycursor.close()
        if mydb:
            mydb.close()

def get_vessel_from_db(mmsi):
    """Henter skipsinformasjon fra databasen basert på MMSI, inkludert ship_type_code."""
    mydb = connect_db()
    if mydb is None:
        return None
    mycursor = mydb.cursor()
    try:
        mycursor.execute(
            # La til ship_type_code i SELECT
            f"SELECT mmsi, callsign, ship_name, ship_type_code, ship_type, image_url FROM {SHIP_TABLE} WHERE mmsi = %s",
            (str(mmsi),),
        )
        result = mycursor.fetchone()
        if result:
            return {
                "mmsi": result[0],
                "callsign": result[1],
                "ship_name": result[2],
                "ship_type_code": result[3], # Nytt felt
                "ship_type": result[4],      # Dette er tekstbeskrivelsen
                "image_url": result[5],
            }
        return None
    except mysql.connector.Error as err:
        print(f"❌ Feil ved henting av fartøydata for MMSI {mmsi}: {err}")
        return None
    finally:
        if mycursor:
            mycursor.close()
        if mydb:
            mydb.close()

def get_ship_type_description(ship_type_code):
    """Henter beskrivelsen av skipstypen basert på tallkoden."""
    ship_types = {
        20: "Wing In Ground (WIG), all ships of this type", 21: "Wing In Ground (WIG), Hazardous category A",
        22: "Wing In Ground (WIG), Hazardous category B", 23: "Wing In Ground (WIG), Hazardous category C",
        24: "Wing In Ground (WIG), Hazardous category D", 25: "Wing In Ground (WIG), Reserved for future use",
        26: "Wing In Ground (WIG), Reserved for future use", 27: "Wing In Ground (WIG), Reserved for future use",
        28: "Wing In Ground (WIG), Reserved for future use", 29: "Wing In Ground (WIG), No additional information",
        30: "Fishing", 31: "Towing", 32: "Towing: length exceeds 200m or breadth exceeds 25m",
        33: "Dredging or underwater ops", 34: "Diving ops", 35: "Military ops", 36: "Sailing",
        37: "Pleasure Craft", 38: "Reserved", 39: "Reserved",
        40: "High speed craft (HSC), all ships of this type", 41: "High speed craft (HSC), Hazardous category A",
        42: "High speed craft (HSC), Hazardous category B", 43: "High speed craft (HSC), Hazardous category C",
        44: "High speed craft (HSC), Hazardous category D", 45: "High speed craft (HSC), Reserved for future use",
        46: "High speed craft (HSC), Reserved for future use", 47: "High speed craft (HSC), Reserved for future use",
        48: "High speed craft (HSC), Reserved for future use", 49: "High speed craft (HSC), No additional information",
        50: "Pilot Vessel", 51: "Search and Rescue vessel", 52: "Tug", 53: "Port Tender",
        54: "Anti-pollution equipment", 55: "Law Enforcement", 56: "Spare - Local Vessel",
        57: "Spare - Local Vessel", 58: "Medical Transport", 59: "Noncombatant ship according to RR Resolution No. 18",
        60: "Passenger, all ships of this type", 61: "Passenger, Hazardous category A",
        62: "Passenger, Hazardous category B", 63: "Passenger, Hazardous category C",
        64: "Passenger, Hazardous category D", 65: "Passenger, Reserved for future use",
        66: "Passenger, Reserved for future use", 67: "Passenger, Reserved for future use",
        68: "Passenger, Reserved for future use", 69: "Passenger, No additional information",
        70: "Cargo, all ships of this type", 71: "Cargo, Hazardous category A", 72: "Cargo, Hazardous category B",
        73: "Cargo, Hazardous category C", 74: "Cargo, Hazardous category D", 75: "Cargo, Reserved for future use",
        76: "Cargo, Reserved for future use", 77: "Cargo, Reserved for future use", 78: "Cargo, Reserved for future use",
        79: "Cargo, No additional information",
        80: "Tanker, all ships of this type", 81: "Tanker, Hazardous category A", 82: "Tanker, Hazardous category B",
        83: "Tanker, Hazardous category C", 84: "Tanker, Hazardous category D", 85: "Tanker, Reserved for future use",
        86: "Tanker, Reserved for future use", 87: "Tanker, Reserved for future use", 88: "Tanker, Reserved for future use",
        89: "Tanker, No additional information",
        90: "Other Type, all ships of this type", 91: "Other Type, Hazardous category A",
        92: "Other Type, Hazardous category B", 93: "Other Type, Hazardous category C",
        94: "Other Type, Hazardous category D", 95: "Other Type, Reserved for future use",
        96: "Other Type, Reserved for future use", 97: "Other Type, Reserved for future use",
        98: "Other Type, Reserved for future use", 99: "Other Type, no additional information"
    }
    return ship_types.get(ship_type_code, f"Ukjent ({ship_type_code})")


def create_live_ais_data_table():
    """Oppretter tabellen 'live_ais_data' hvis den ikke allerede eksisterer,
       inkludert en kolonne for 'additional_info'."""
    mydb = connect_db()
    if mydb is None:
        return
    mycursor = mydb.cursor()
    table_name = "live_ais_data" 
    try:
        mycursor.execute(f"SHOW COLUMNS FROM `{table_name}` LIKE 'additional_info';")
        column_exists = mycursor.fetchone()

        if not column_exists:
            try:
                mycursor.execute(f"ALTER TABLE `{table_name}` ADD COLUMN additional_info TEXT NULL AFTER ais_message_type;")
                mydb.commit()
                print(f"✅ Kolonnen 'additional_info' lagt til i tabellen '{table_name}'.")
            except mysql.connector.Error as alter_err:
                print(f"ℹ️ Kunne ikke legge til kolonnen 'additional_info' med ALTER TABLE (feil: {alter_err}). Prøver CREATE TABLE IF NOT EXISTS.")

        mycursor.execute(
            f"""
            CREATE TABLE IF NOT EXISTS `{table_name}` (
                mmsi VARCHAR(15) PRIMARY KEY,
                latitude DECIMAL(9,6) NOT NULL,
                longitude DECIMAL(9,6) NOT NULL,
                speed_knots DECIMAL(5,1) NULL,
                course_deg DECIMAL(5,1) NULL,
                true_heading_deg INT NULL,
                nav_status_code INT NULL,
                rate_of_turn INT NULL,
                ais_message_type INT NULL,
                additional_info TEXT NULL,
                last_update_time TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (mmsi) REFERENCES {SHIP_TABLE}(mmsi) ON DELETE CASCADE ON UPDATE CASCADE
            )
            """
        )
        mydb.commit()
        print(f"✅ Tabell '{table_name}' er klar (opprettet eller eksisterer allerede med nødvendige kolonner).")
    except mysql.connector.Error as err:
        print(f"❌ Feil ved klargjøring av tabell '{table_name}': {err}")
        traceback.print_exc()
    finally:
        if 'mycursor' in locals() and mycursor:
            mycursor.close()
        if 'mydb' in locals() and mydb and mydb.is_connected():
            mydb.close()


def update_db(mmsi, ais_data):
    """
    Oppdaterer skipsinformasjon i databasen.
    Lagrer numerisk skipstypekode og tekstbeskrivelse av skipstypen.
    Felter oppdateres kun hvis ny, ikke-null data mottas fra AIS-meldingen.
    """
    mydb = connect_db()
    if mydb is None:
        print(f"FEIL [update_db]: Kunne ikke koble til database for MMSI {mmsi}. Avbryter DB-oppdatering.")
        return
    
    mycursor = mydb.cursor()
    try:
        # 'ais_data.get("ship_type")' forventes å være den numeriske koden fra AIS-meldingen (f.eks. msg 5, 24)
        ship_type_numeric_code = ais_data.get("ship_type") # Dette er en int eller None
        ship_type_text_description = None

        if isinstance(ship_type_numeric_code, int):
            ship_type_text_description = get_ship_type_description(ship_type_numeric_code)
        # Hvis ship_type_numeric_code er None (f.eks. fra en posisjonsrapport uten typeinfo),
        # vil ship_type_text_description også være None.
        # COALESCE i SQL vil da beholde eksisterende verdier i databasen.

        sql = f"""
        INSERT INTO {SHIP_TABLE} (mmsi, callsign, ship_name, ship_type_code, ship_type, last_updated)
        VALUES (%s, %s, %s, %s, %s, %s)
        ON DUPLICATE KEY UPDATE
            callsign = COALESCE(VALUES(callsign), callsign),
            ship_name = COALESCE(VALUES(ship_name), ship_name),
            ship_type_code = COALESCE(VALUES(ship_type_code), ship_type_code),
            ship_type = COALESCE(VALUES(ship_type), ship_type),
            last_updated = VALUES(last_updated)
        """
        
        # Hent verdier fra ais_data. Hvis en nøkkel ikke finnes, vil .get() returnere None.
        # COALESCE vil da sørge for at eksisterende DB-verdi beholdes.
        val_callsign = ais_data.get("callsign")
        val_name = ais_data.get("name")
        # ship_type_numeric_code og ship_type_text_description er allerede definert over

        now_utc_str = datetime.now(timezone.utc).strftime('%Y-%m-%d %H:%M:%S')
        values = (
                str(mmsi),
                val_callsign,
                val_name,
                ship_type_numeric_code,
                ship_type_text_description,
                now_utc_str
        )
        
        mycursor.execute(sql, values)
        mydb.commit()

        # Forbedret logging
        if mycursor.rowcount == 1: # Ny rad satt inn
            action_details = ["Ny rad satt inn"]
            if val_callsign: action_details.append(f"kallesignal='{val_callsign}'")
            if val_name: action_details.append(f"navn='{val_name}'")
            if ship_type_numeric_code is not None:
                action_details.append(f"skipstypekode={ship_type_numeric_code}")
                action_details.append(f"skipstype='{ship_type_text_description}'")
            print(f"✅ [DB] MMSI {mmsi}: {', '.join(action_details)}.")
        elif mycursor.rowcount == 2: # Eksisterende rad ble oppdatert (endret)
            action_details = ["Rad oppdatert"]
            # Logg kun felter som faktisk KUNNE ha blitt oppdatert av denne meldingen
            updated_fields = []
            if val_callsign is not None: updated_fields.append(f"kallesignal='{val_callsign}'")
            if val_name is not None: updated_fields.append(f"navn='{val_name}'")
            if ship_type_numeric_code is not None:
                 updated_fields.append(f"skipstypekode={ship_type_numeric_code}")
                 updated_fields.append(f"skipstype='{ship_type_text_description}'")
            
            if updated_fields:
                action_details.append("med data: " + ", ".join(updated_fields))
            else:
                action_details.append("(ingen nye data mottatt for statiske felter i denne meldingen, kun last_updated endret hvis raden fantes).")
            print(f"✅ [DB] MMSI {mmsi}: {', '.join(action_details)}.")
        elif mycursor.rowcount == 0: # Rad fantes, men ingen verdier ble endret av UPDATE (COALESCE beholdt alt)
            print(f"ℹ️ [DB] MMSI {mmsi}: Forsøkte oppdatering, men ingen verdier endret seg (COALESCE). `last_updated` er heller ikke endret.")
            
    except mysql.connector.Error as err:
        print(f"❌ [DB] Feil ved databaseoperasjon for MMSI {mmsi}: {err}")
        print(f"   SQL: {mycursor.statement if mycursor else 'N/A'}") # Logg SQL-setningen
        print(f"   Verdier: {values if 'values' in locals() else 'N/A'}") # Logg verdiene
        if mydb and mydb.is_connected(): 
            try:
                mydb.rollback()
                print("ℹ️ [DB] Rollback utført.")
            except mysql.connector.Error as rb_err:
                print(f"⚠️ [DB] Feil under rollback: {rb_err}")
    except Exception as e:
        print(f"❌ [DB] Uventet feil i update_db for MMSI {mmsi}: {e}")
        traceback.print_exc()
    finally:
        if mycursor:
            mycursor.close()
        if mydb and mydb.is_connected(): 
            mydb.close()

def update_json_file(ais_data):
    """
    Oppdaterer JSON-filen med AIS-data (ship_type som TEKST),
    rydder opp gamle skip basert på timeout.
    """
    try:
        processed_data = {}
        for key, value in ais_data.items(): 
            if isinstance(value, bytes):
                processed_data[key] = value.decode(errors="ignore")
            elif isinstance(value, (NavigationStatus, ManeuverIndicator, EpfdType, NavAid, TurnRate)):
                processed_data[key] = value.name
            else:
                processed_data[key] = value
        
        mmsi = processed_data.get("mmsi")
        lat = processed_data.get("lat")
        lon = processed_data.get("lon")

        if not mmsi or lat is None or lon is None:
            return

        vessel_info = get_vessel_from_db(mmsi) # Henter nå også ship_type_code
        ship_name_from_db = vessel_info.get("ship_name") if vessel_info else None
        
        # Bruk navn fra meldingen hvis tilgjengelig og gyldig, ellers fra DB, ellers "Ukjent"
        ship_name_from_message = processed_data.get("name", "").strip()
        if ship_name_from_message and ship_name_from_message.upper() != "N/A" and not all(c == "@" for c in ship_name_from_message):
            final_ship_name = ship_name_from_message
        elif ship_name_from_db:
            final_ship_name = ship_name_from_db
        else:
            final_ship_name = "Ukjent"

        ship_type_code_from_current_ais_message = processed_data.get("ship_type") 
        ship_type_desc_from_db = vessel_info.get("ship_type") if vessel_info else None # Tekstbeskrivelse fra DB
        
        final_ship_type_description = "Ukjent" 

        if isinstance(ship_type_code_from_current_ais_message, int):
            final_ship_type_description = get_ship_type_description(ship_type_code_from_current_ais_message)
        elif ship_type_desc_from_db is not None:
            final_ship_type_description = ship_type_desc_from_db
        
        current_time_utc = datetime.now(timezone.utc)
        ship_data_for_json = {
            "mmsi": mmsi, "lat": lat, "lon": lon, "ship_name": final_ship_name,
            "ship_type": final_ship_type_description, "msg_type": processed_data.get("msg_type"),
            "speed": processed_data.get("speed"), "course": processed_data.get("course"),
            "heading": processed_data.get("heading"),
            "timestamp": current_time_utc.strftime('%Y-%m-%dT%H:%M:%SZ')
        }
        ship_data_for_json = {k: v for k, v in ship_data_for_json.items() if v is not None}

        ships_data = {"ships": []}
        if os.path.exists(JSON_FILE):
            try:
                with open(JSON_FILE, "r", encoding="utf-8") as f:
                    file_content = f.read()
                    if file_content:
                        loaded_data = json.loads(file_content)
                        if isinstance(loaded_data, dict) and "ships" in loaded_data and isinstance(loaded_data["ships"], list):
                            ships_data = loaded_data
                        else:
                            print(f"⚠️ JSON-fil {JSON_FILE} har uventet format, starter med tom liste.")
            except json.JSONDecodeError:
                print(f"⚠️ Kunne ikke dekode JSON fra {JSON_FILE}, starter med tom liste.")
            except Exception as read_err:
                print(f"❌ Uventet feil ved lesing av JSON {JSON_FILE}: {read_err}")
                traceback.print_exc()

        updated_ships_list = []
        now_utc = datetime.now(timezone.utc)
        max_age = timedelta(minutes=SHIP_TIMEOUT_MINUTES)
        for ship in ships_data.get("ships", []):
            try:
                timestamp_str = ship.get("timestamp")
                if timestamp_str:
                    if timestamp_str.endswith('Z'):
                        timestamp_str = timestamp_str[:-1] + '+00:00'
                    ship_timestamp = datetime.fromisoformat(timestamp_str)
                    if now_utc - ship_timestamp <= max_age:
                        updated_ships_list.append(ship)
            except (ValueError, TypeError, AttributeError) as ts_err:
                print(f"⚠️ Kunne ikke behandle timestamp for skip {ship.get('mmsi', 'UKJENT')}: {ts_err}. Fjerner fra listen.")
                continue

        found_in_cleaned_list = False
        for i, existing_ship in enumerate(updated_ships_list):
            try:
                if str(existing_ship.get("mmsi")) == str(mmsi):
                    updated_ships_list[i] = ship_data_for_json
                    found_in_cleaned_list = True
                    break
            except AttributeError:
                 print(f"⚠️ Feil format på skip-objekt under oppdatering: {existing_ship}")
                 continue
        if not found_in_cleaned_list:
            updated_ships_list.append(ship_data_for_json)

        final_data_to_write = {"ships": updated_ships_list}
        
        try:
            temp_file = JSON_FILE + ".tmp"
            with open(temp_file, "w", encoding="utf-8") as f:
                json.dump(final_data_to_write, f, indent=2, ensure_ascii=False)
            os.replace(temp_file, JSON_FILE)
            # print(f"✅ [JSON] Fil {JSON_FILE} oppdatert med data for MMSI {mmsi}. Totalt antall skip i fil: {len(updated_ships_list)}.")

        except Exception as write_err:
            print(f"❌❌❌ FEIL VED SKRIVING/REPLACE AV JSON ({JSON_FILE}) for MMSI {mmsi}: {type(write_err).__name__} - {write_err}")
            traceback.print_exc()

    except Exception as e:
        mmsi_for_log = processed_data.get('mmsi') if 'processed_data' in locals() and isinstance(processed_data, dict) else (ais_data.get('mmsi') if isinstance(ais_data, dict) else 'UKJENT')
        print(f"❌ Generell feil i update_json_file for MMSI {mmsi_for_log}: {e}")
        traceback.print_exc()


def update_live_ais_data_in_db(ais_data):
    if not isinstance(ais_data, dict):
        print(f"FEIL [update_live_db]: ais_data er ikke en dictionary. Avbryter.")
        return

    mmsi_val = str(ais_data.get("mmsi"))
    lat_val = ais_data.get("lat")
    lon_val = ais_data.get("lon")

    if not mmsi_val or lat_val is None or lon_val is None:
        return

    mydb = connect_db()
    if mydb is None:
        return
    
    mycursor = mydb.cursor()
    try:
        # STEG 1: HENT ALLE DATA FRA AIS-MELDINGEN (som i din originale kode)
        explicit_keys = {'mmsi', 'lat', 'lon', 'speed', 'course', 'heading', 'status', 'turn', 'msg_type'}
        other_info_dict = {}
        for key, value in ais_data.items():
            if key not in explicit_keys:
                if isinstance(value, (NavigationStatus, ManeuverIndicator, EpfdType, NavAid, TurnRate)):
                    other_info_dict[key] = value.name
                elif isinstance(value, bytes):
                    other_info_dict[key] = value.decode(errors='ignore')
                else:
                    other_info_dict[key] = value
        
        additional_info_json = json.dumps(other_info_dict) if other_info_dict else None
        nav_status_obj = ais_data.get('status')
        nav_status_code = nav_status_obj.value if nav_status_obj is not None else None
        rot_obj = ais_data.get('turn')
        rate_of_turn_val = rot_obj.value if isinstance(rot_obj, TurnRate) else (rot_obj if isinstance(rot_obj, int) else None)

        # STEG 2: LAG ET KORREKT TIDSSTEMPEL FRA DIN PI
        now_utc_str = datetime.now(timezone.utc).strftime('%Y-%m-%d %H:%M:%S')

        # STEG 3: BRUK EN SQL-SPØRRING SOM SENDER MED DET KORREKTE TIDSSTEMPELET
        sql_live = """
            INSERT INTO live_ais_data (
                mmsi, latitude, longitude, speed_knots, course_deg, 
                true_heading_deg, nav_status_code, rate_of_turn, ais_message_type,
                additional_info, last_update_time
            ) VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)
            ON DUPLICATE KEY UPDATE
                latitude = VALUES(latitude),
                longitude = VALUES(longitude),
                speed_knots = VALUES(speed_knots),
                course_deg = VALUES(course_deg),
                true_heading_deg = VALUES(true_heading_deg),
                nav_status_code = VALUES(nav_status_code),
                rate_of_turn = VALUES(rate_of_turn),
                ais_message_type = VALUES(ais_message_type),
                additional_info = VALUES(additional_info),
                last_update_time = VALUES(last_update_time)
        """
        values_live = (
            mmsi_val, lat_val, lon_val,
            ais_data.get('speed'), ais_data.get('course'), ais_data.get('heading'),
            nav_status_code, rate_of_turn_val, ais_data.get('msg_type'),
            additional_info_json,
            now_utc_str  # Sender med det korrekte tidsstempelet
        )

        mycursor.execute(sql_live, values_live)
        mydb.commit()
        print(f"✅ [DB_LIVE] Lagret posisjon for MMSI {mmsi_val} med korrekt tid.")

    except mysql.connector.Error as err:
        print(f"❌ [DB_LIVE] Feil ved oppdatering av live data for MMSI {mmsi_val}: {err}")
    finally:
        if mydb.is_connected():
            mycursor.close()
            mydb.close()

def parse_ais_data(raw_message):
    """
    Parser en rå AIS NMEA-streng.
    1. Kaller update_db for å sikre at MMSI er i 'ships'-tabellen og oppdaterer 
       eventuell statisk info (navn, kallesignal, skipstype (numerisk og tekst)) fra meldingen.
    2. Hvis posisjonsdata finnes:
       a. Kaller update_live_ais_data_in_db for å lagre dynamiske data.
       b. Returnerer en dictionary med meldingsdata (for JSON-oppdatering).
    3. Returnerer None hvis ingen posisjonsdata eller ved feil.
    """
    try:
        if raw_message.startswith("!AIVDM") or raw_message.startswith("!AIVDO"):
            decoded = decode(raw_message)
            decoded_dict = decoded.asdict() 

            if hasattr(decoded, 'message_id') and decoded.message_id == 5:
                mmsi_val = str(decoded.mmsi)
                static_payload_for_db = {}
                shipname_msg5 = getattr(decoded, 'shipname', None)
                callsign_msg5 = getattr(decoded, 'callsign', None)
                shiptype_msg5 = getattr(decoded, 'shiptype', None) # Numerisk kode

                if shipname_msg5 is not None:
                    cleaned_name = shipname_msg5.strip().rstrip('@') # Fjern trailing @
                    if cleaned_name and cleaned_name.upper() != "N/A":
                        static_payload_for_db["name"] = cleaned_name
                if callsign_msg5 is not None:
                    cleaned_callsign = callsign_msg5.strip().rstrip('@')
                    if cleaned_callsign and cleaned_callsign.upper() != "N/A":
                        static_payload_for_db["callsign"] = cleaned_callsign
                if shiptype_msg5 is not None: # Kan være 0, som er gyldig ("Not available")
                    static_payload_for_db["ship_type"] = shiptype_msg5 # Send numerisk kode til update_db
                
                if static_payload_for_db: 
                    # print(f"🚢 Mottok Melding 5 for MMSI {mmsi_val} med data: {static_payload_for_db}")
                    update_db(mmsi_val, static_payload_for_db)
                return None # Melding 5 har ikke posisjon for JSON/live_ais_data via denne funksjonens retur

            elif "mmsi" in decoded_dict and decoded_dict.get("mmsi") is not None:
                mmsi_val = str(decoded_dict.get("mmsi"))
                static_payload_for_db = {}
                
                # Melding 24A/B kan ha navn, type, kallesignal
                shipname_from_msg = decoded_dict.get("shipname") 
                callsign_from_msg = decoded_dict.get("callsign")
                shiptype_from_msg = decoded_dict.get("shiptype") # Numerisk kode

                if shipname_from_msg is not None:
                    cleaned_name = shipname_from_msg.strip().rstrip('@')
                    if cleaned_name and cleaned_name.upper() != "N/A":
                        static_payload_for_db["name"] = cleaned_name
                if callsign_from_msg is not None:
                    cleaned_callsign = callsign_from_msg.strip().rstrip('@')
                    if cleaned_callsign and cleaned_callsign.upper() != "N/A":
                        static_payload_for_db["callsign"] = cleaned_callsign
                if shiptype_from_msg is not None:
                    static_payload_for_db["ship_type"] = shiptype_from_msg # Send numerisk kode

                # Kall update_db for å sikre at MMSI eksisterer og oppdatere evt. statisk info.
                # Selv om static_payload_for_db er tom, vil kallet sikre at MMSI-raden finnes
                # (hvis den ikke gjør det, vil den bli opprettet med None for de andre feltene).
                # COALESCE i update_db håndterer at eksisterende data ikke overskrives med None.
                update_db(mmsi_val, static_payload_for_db) 
                
                if "lon" in decoded_dict and decoded_dict.get("lon") is not None and \
                   "lat" in decoded_dict and decoded_dict.get("lat") is not None:
                    try:
                        decoded_dict["lon"] = float(decoded_dict["lon"])
                        decoded_dict["lat"] = float(decoded_dict["lat"])
                        update_live_ais_data_in_db(decoded_dict) 
                        return decoded_dict 
                    except (ValueError, TypeError) as e:
                        print(f"⚠️ Feil [parse_ais_data]: Kunne ikke konvertere lon/lat til float for MMSI {mmsi_val}: lon={decoded_dict.get('lon')}, lat={decoded_dict.get('lat')}. Feil: {e}")
                        return None
                else:
                    return None
            else:
                return None
        else:
            return None
            
    except MissingMultipartMessageException as e:
        # Disse er vanlige og ikke nødvendigvis en "feil" hvis man ikke venter på alle deler
        # print(f"ℹ️ Manglende deler for melding (vanlig): '{raw_message[:70]}...' Feil: {e}")
        return None
    except Exception as e: 
        print(f"❌ Uventet feil under parsing av melding: '{raw_message[:70]}...' Feil: {e} ({type(e).__name__})")
        traceback.print_exc()
        return None


def read_udp_ais_data():
    UDP_IP = "127.0.0.1" 
    listen_sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    try:
        listen_sock.bind((UDP_IP, UDP_PORT))
        print(f"🎧 Lytter etter AIS data på UDP {UDP_IP}:{UDP_PORT}")
    except socket.error as e:
        print(f"❌ Kunne ikke binde til UDP port {UDP_PORT}: {e}")
        print("   Sjekk om en annen prosess bruker porten, eller kjør med sudo hvis nødvendig.")
        raise RuntimeError(f"Kunne ikke binde til UDP port {UDP_PORT}: {e}") # Stopp skriptet

    aishub_sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    # print(f"📡 Klar til å videresende rådata til AISHub {AISHUB_IP}:{AISHUB_PORT}") # Mindre støy

    try:
        while True:
            try:
                data, addr = listen_sock.recvfrom(1024) 
                if data:
                    try:
                        aishub_sock.sendto(data, (AISHUB_IP, AISHUB_PORT))
                    except socket.error as se_aishub:
                        print(f"❌ Socket-feil ved sending til AISHub: {se_aishub}")
                    except Exception as e_aishub:
                        print(f"❌ Uventet feil ved sending til AISHub: {e_aishub} ({type(e_aishub).__name__})")
                        # traceback.print_exc() # Vurder ved feilsøking

                raw_message = data.decode("utf-8", errors="ignore").strip()
                if not raw_message: 
                    continue

                ais_data = parse_ais_data(raw_message) # Denne kaller update_db og update_live_ais_data_in_db

                if ais_data: # ais_data er dict hvis posisjonsdata var gyldig
                    update_json_file(ais_data)
                
            except socket.timeout:
                # print(f"DEBUG [read_udp_ais_data]: Socket timeout på listen_sock, fortsetter å lytte...")
                continue 
            except socket.error as e_sock:
                print(f"❌ Socket-feil ved mottak av UDP-data (lokal lytting): {e_sock} ({type(e_sock).__name__})")
                traceback.print_exc()
                time.sleep(1) 
                continue 
            except Exception as e_loop: 
                print(f"❌ Uventet feil i read_udp_ais_data-hovedløkken: {e_loop} ({type(e_loop).__name__})")
                traceback.print_exc() 
                time.sleep(5) 

    except KeyboardInterrupt:
        print("\n👋 Avbrutt av bruker (Ctrl+C).")
    finally:
        print("🔌 Lukker sockets...")
        if 'listen_sock' in locals() and listen_sock:
            listen_sock.close()
        if 'aishub_sock' in locals() and aishub_sock:
            aishub_sock.close()


def start_rtl_ais():
    rtl_ais_cmd = ["/home/pi/rtl-ais/rtl_ais", "-n", "-d", "0", "-P", str(UDP_PORT)]
    try:
        print(f"🚀 Starter rtl_ais med kommando: {' '.join(rtl_ais_cmd)}")
        process = subprocess.Popen(rtl_ais_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        print(f"✅ rtl_ais startet med PID: {process.pid}")
        time.sleep(2) 
        if process.poll() is not None:
             stderr_output = process.stderr.read().decode(errors='ignore')
             print(f"❌ rtl_ais avsluttet uventet rett etter start. Feilmelding:")
             print(stderr_output)
             return None 
        return process
    except FileNotFoundError:
        print(f"❌ Kommando '{rtl_ais_cmd[0]}' ikke funnet. Sørg for at rtl_ais er installert og stien er korrekt.")
        return None
    except Exception as e:
        print(f"❌ Ukjent feil ved start av rtl_ais: {e}")
        return None


def free_udp_port(port):
    print(f"🔎 Sjekker om UDP-port {port} er i bruk...")
    pid_found = None
    try:
        result = subprocess.run(["ss", "-lupn"], capture_output=True, text=True, check=False)
        for line in result.stdout.splitlines():
            if f":{port}" in line and "users:" in line:
                try:
                    pid_part = line.split("users:")[1]
                    if "pid=" in pid_part:
                        pid_str = pid_part.split("pid=")[1].split(",")[0]
                        pid_found = int(pid_str)
                        print(f"⚠️ Port {port} er i bruk av prosess med PID {pid_found}.")
                        break 
                except (IndexError, ValueError) as parse_err:
                    print(f"⚠️ Kunne ikke parse PID fra ss output linje: {line}. Feil: {parse_err}")
                    continue 
    except FileNotFoundError:
        print("⚠️ Kommandoen 'ss' ble ikke funnet. Kan ikke sjekke portbruk automatisk. Prøv 'sudo apt install iproute2'.")
        return
    except Exception as e:
        print(f"⚠️ Feil under kjøring av 'ss' kommando: {e}")
        return

    if pid_found:
        print(f"🛑 Prøver å avslutte prosess {pid_found} som bruker port {port}...")
        try:
            os.kill(pid_found, signal.SIGTERM)
            time.sleep(2) 
            if psutil.pid_exists(pid_found):
                print(f"🛑 Prosess {pid_found} avsluttet ikke med SIGTERM, prøver SIGKILL...")
                os.kill(pid_found, signal.SIGKILL)
                time.sleep(1) 
                if psutil.pid_exists(pid_found):
                    print(f"❌ Klarte ikke å avslutte prosess {pid_found} med SIGKILL.")
                    raise RuntimeError(f"Kan ikke frigjøre port {port} fra PID {pid_found}")
                else:
                    print(f"✅ Prosess {pid_found} tvunget til å avslutte (SIGKILL).")
            else:
                 print(f"✅ Prosess {pid_found} avsluttet (SIGTERM).")
            time.sleep(1)
            with socket.socket(socket.AF_INET, socket.SOCK_DGRAM) as s:
                try:
                    s.bind(("0.0.0.0", port))
                    print(f"✅ Port {port} er nå ledig.")
                except OSError:
                    print(f"❌ Port {port} er fortsatt i bruk etter forsøk på å drepe PID {pid_found}.")
                    raise RuntimeError(f"Klarte ikke å frigjøre port {port}")
        except ProcessLookupError:
            print(f"ℹ️ Prosess {pid_found} fantes ikke (kanskje den avsluttet seg selv).")
            with socket.socket(socket.AF_INET, socket.SOCK_DGRAM) as s:
                try: s.bind(("0.0.0.0", port)); print(f"✅ Port {port} er nå ledig.")
                except OSError:
                    print(f"❌ Port {port} er fortsatt i bruk selv om PID {pid_found} ikke finnes."); raise RuntimeError(f"Kan ikke binde til port {port}")
        except PermissionError:
             print(f"❌ Ingen rettigheter til å avslutte prosess {pid_found}. Prøv å kjøre scriptet med sudo."); raise 
        except Exception as e:
            print(f"⚠️ Uventet feil under avslutning av prosess {pid_found}: {e}"); raise
    else:
         print(f"✅ Port {port} ser ut til å være ledig.")

# --- Hovedprogram ---
if __name__ == "__main__":
    was_manual_run = not is_running_as_systemd_service()

    if was_manual_run:
        print("INFO: Detektert manuell kjøring (ikke som systemd service).")
        print("INFO: Prøver å stoppe en eventuell eksisterende ais_to_json.service...")
        try:
            stop_command = ["sudo", "systemctl", "stop", "ais_to_json.service"]
            result = subprocess.run(stop_command, capture_output=True, text=True, timeout=15, check=False)
            if result.returncode == 0:
                print("INFO: Stopp-kommando for ais_to_json.service sendt. Venter 5 sekunder...")
                time.sleep(5)
            elif any(msg in result.stderr.lower() for msg in ["unit ais_to_json.service not loaded", "could not be found", "no such unit", "unit ais_to_json.service not found"]):
                print("INFO: ais_to_json.service var ikke lastet/funnet (sannsynligvis ikke aktiv).")
            else:
                print(f"ADVARSEL: 'sudo systemctl stop ais_to_json.service' returnerte kode {result.returncode}.")
                if result.stdout.strip(): print(f"STDOUT: {result.stdout.strip()}")
                if result.stderr.strip(): print(f"STDERR: {result.stderr.strip()}")
        except subprocess.TimeoutExpired: print("ADVARSEL: Tidsavbrudd under forsøk på å stoppe ais_to_json.service.")
        except FileNotFoundError: print("ADVARSEL: 'sudo' eller 'systemctl' kommando ikke funnet.")
        except Exception as e: print(f"ADVARSEL: Uventet feil under forsøk på å stoppe ais_to_json.service: {e}")
    else:
        print("INFO: Kjører som systemd service. Hopper over 'sudo systemctl stop ais_to_json.service'.")

    rtl_ais_process = None
    try:
        # STEG 1: Start det lokale først
        print("🔄 Sjekker og frigjør UDP-port...")
        free_udp_port(UDP_PORT)
        
        rtl_ais_process = start_rtl_ais()
        if rtl_ais_process is None:
             exit(1)

        # STEG 2: Sjekk databasen ETTER at det lokale kjører
        create_ship_table() 
        create_live_ais_data_table()
        
        # STEG 3: Start hovedløkken for lytting
        read_udp_ais_data()

    except KeyboardInterrupt:
        print("\n🚦 Avbrudd mottatt (Ctrl+C). Avslutter...")
        if was_manual_run:
            print("--------------------------------------------------------------------")
            print("ℹ️ For å starte tjenesten i bakgrunnen igjen, bruk kommandoen:")
            print("   sudo systemctl start ais_to_json.service")
            print("--------------------------------------------------------------------")
            
    except RuntimeError as e: # Fanger kritiske feil som krever avslutning (f.eks. portbinding)
        print(f"❌ Kritisk feil oppstod: {e}. Avslutter.")
        traceback.print_exc()
        exit(1) # Avslutt med feilkode
    except Exception as e:
         print(f"❌ En uventet feil oppstod i hovedprogrammet: {e}")
         traceback.print_exc()

    finally:
        if rtl_ais_process and rtl_ais_process.poll() is None:
            print("🧹 Stopper rtl_ais prosessen...")
            try:
                rtl_ais_process.terminate()
                rtl_ais_process.wait(timeout=5)
                print("✅ rtl_ais stoppet (SIGTERM).")
            except subprocess.TimeoutExpired:
                print("⚠️ rtl_ais svarte ikke på SIGTERM, sender SIGKILL...")
                rtl_ais_process.kill()
                try:
                    rtl_ais_process.wait(timeout=2)
                    print("✅ rtl_ais tvunget til å stoppe (SIGKILL).")
                except subprocess.TimeoutExpired:
                    print("⚠️ rtl_ais svarte ikke på SIGKILL innen timeout.")
            except Exception as e:
                print(f"⚠️ Feil under stopping av rtl_ais: {e}")
        elif rtl_ais_process:
             print("ℹ️ rtl_ais prosessen var allerede stoppet.")
        else:
             print("ℹ️ Ingen rtl_ais prosess å stoppe (ble kanskje aldri startet).")

        print("🏁 Program avsluttet.")
