# -*- coding: utf-8 -*-
import mysql.connector # <-- MySQL endring: Importerer MySQL-biblioteket
from mysql.connector import Error as MySQLError 
# <-- MySQL endring: For feilhåndtering
import time
import re
import os
import requests
import random
import json
import sys
from datetime import datetime, timedelta # <--- NYTT: Importer datetime

from decimal import Decimal, InvalidOperation

# duckduckgo_search trengs fortsatt for søkefunksjonen
from duckduckgo_search import DDGS

# BeautifulSoup trengs fortsatt for Brave-søk (hvis installert)
try:
    from bs4 import BeautifulSoup
except ImportError:
    print("⚠️ Advarsel: BeautifulSoup (bs4) er ikke installert. HTML-parsing i Brave-søk vil feile.")
    # Definer en dummy-klasse hvis bs4 mangler
    class DummySoup:
        def __init__(self, *args, **kwargs): pass
        def get_text(self, *args, **kwargs): return ""
        def finditer(self, *args, **kwargs): return []
    BeautifulSoup = DummySoup

from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC

from dotenv import load_dotenv
env_path = '/home/pi/key_info.env'
load_dotenv(dotenv_path=env_path)

def search_kystverket_image(mmsi):
    """Bruker en headless nettleser (Selenium) for å hente bilde-URL fra Kystverket."""
    print(f"  📸 Søker etter bilde på Kystverket for MMSI {mmsi} med Selenium...")
    
    chrome_options = Options()
    chrome_options.add_argument("--headless")
    chrome_options.add_argument("--no-sandbox")
    chrome_options.add_argument("--disable-dev-shm-usage")
    
    driver = None
    try:
        driver = webdriver.Chrome(options=chrome_options)
        url = f"https://nais.kystverket.no/aistream/{mmsi}"
        driver.get(url)
        
        # Vent i opptil 15 sekunder på at bilde-elementet skal bli synlig
        # Vi bruker CSS-selektoren fra bildet ditt: 'img.vessel-photo'
        wait = WebDriverWait(driver, 15)
        img_element = wait.until(EC.presence_of_element_located((By.CSS_SELECTOR, "img.vessel-photo")))
        
        image_url = img_element.get_attribute('src')
        
        if image_url:
            print(f"  ✅ Fant bilde-URL fra Kystverket: {image_url}")
            return {"image_url": image_url}
        else:
            print(f"  ℹ️ Fant bilde-elementet, men det hadde ingen src-URL.")
            return None
            
    except Exception as e:
        print(f"  ⚠️ Feil under Selenium-søk for MMSI {mmsi}: {e}")
        return None
    finally:
        if driver:
            driver.quit()


# --- Konfigurasjon (Oppdatert for MySQL) ---
SEARCH_INTERVAL_MINUTES = 15
IMAGE_FOLDER = "/var/www/html/shipimage"
AISHUB_USERNAME = 'AH_3505_563BE20B'
AISHUB_API_URL = "https://data.aishub.net/ws.php"
RESET_PAUSE_HOURS = 6
os.makedirs(IMAGE_FOLDER, exist_ok=True)
search_counter = random.choice([0, 1])

# --- MySQL Database Konfigurasjon ---
DB_HOST = os.getenv("DB_HOST")
DB_USER = os.getenv("DB_USER")
DB_PASSWORD = os.getenv("DB_PASSWORD")
DB_NAME = "totlando_aisdata"
SHIP_TABLE = "ships" # Navn på tabellen i databasen

# --- Hjelpefunksjoner (Ingen endringer nødvendig her) ---

# Sett til True for å deaktivere det respektive søket fra starten av skriptkjøringen.
# Skriptet kan fortsatt dynamisk blokkere et søk internt hvis det er aktivert her, men feiler underveis.
START_WITH_BRAVE_BLOCKED = True  # Endre til True for å starte med Brave deaktivert
START_WITH_DDG_BLOCKED = True    # Endre til True for å starte med DuckDuckGo deaktivert


# --- Pausefunksjon ---

def controlled_pause(min_delay_sec=240, max_delay_sec=360, external=True, reset_cycle=False):
    """
    Sørger for en kontrollert pause med detaljert logging.

    - min_delay_sec: Minimum tid å vente (sekunder).
    - max_delay_sec: Maks tid å vente (sekunder).
    - external: Hvis False, ingen pause (interne operasjoner).
    - reset_cycle: Hvis True, pauser for RESET_PAUSE_HOURS.
    """
    global RESET_PAUSE_HOURS
    try:
        reset_hours = RESET_PAUSE_HOURS
    except NameError:
        reset_hours = 6 # Sett en default hvis ikke definert globalt

    if not external:
        return # Ingen pause for interne operasjoner

    if reset_cycle:
        pause_time = reset_hours * 3600
        # Oppdaterer print-meldingen for å reflektere den nye logg-stilen nedenfor
        # print(f"⏳ Full gjennomgang fullført! Pauser i {reset_hours} timer før ny runde...")
        pause_reason_msg = f"Full gjennomgang fullført! Pauser i {reset_hours} timer"
    else:
        # Sikrer at min/max er fornuftige tall
        min_d = float(min_delay_sec) if isinstance(min_delay_sec, (int, float)) else 60.0
        max_d = float(max_delay_sec) if isinstance(max_delay_sec, (int, float)) else 120.0
        if min_d > max_d: # Bytt om hvis rekkefølgen er feil
             min_d, max_d = max_d, min_d
        pause_time = random.uniform(min_d, max_d)
        # Oppdaterer print-meldingen
        # print(f"⏳ Pauser i {pause_time:.1f} sekunder ({pause_time / 60:.2f} minutter)...")
        pause_reason_msg = f"Pauser i {pause_time / 60:.2f} minutter"

    # <--- START NYTT FOR DETALJERT PAUSELOGGING ---
    current_time_start_pause = datetime.now()
    expected_resume_time = current_time_start_pause + timedelta(seconds=pause_time)

    print(f"⏳ [{current_time_start_pause.strftime('%Y-%m-%d %H:%M:%S')}] {pause_reason_msg}.")
    print(f"     ⏱️ Nøyaktig pauselengde: {pause_time:.2f} sekunder.")
    print(f"     🔄 Forventet gjenopptak: {expected_resume_time.strftime('%Y-%m-%d %H:%M:%S')}")
    # <--- SLUTT NYTT FOR DETALJERT PAUSELOGGING ---

    time.sleep(pause_time)

    # <--- START NYTT FOR LOGGING ETTER PAUSE ---
    actual_resume_time = datetime.now()
    print(f"👍 [{actual_resume_time.strftime('%Y-%m-%d %H:%M:%S')}] Kode fortsetter etter pause.")
    # <--- SLUTT NYTT FOR LOGGING ETTER PAUSE ---

# --- Datakonvertering og validering ---


# NYTT: Valideringsfunksjon for IMO
def is_valid_imo(imo_candidate):
    """Sjekker om et IMO-nummer er gyldig (7 siffer, består kun av tall, og ikke bare nuller)."""
    if imo_candidate is None:
        return False
    imo_str = str(imo_candidate).strip()
    if not (imo_str.isdigit() and len(imo_str) == 7):
        return False
    if int(imo_str) == 0: # IMO 0000000 er ikke et gyldig IMO
        return False
    # Valgfri: Implementer full IMO sjekksumvalidering her hvis ønskelig
    # Eksempel (krever korrekt implementasjon):
    # total = 0
    # for i in range(6):
    #     total += int(imo_str[i]) * (7 - i)
    # return total % 10 == int(imo_str[6])
    return True

# NYTT: Valideringsfunksjon for DRAUGHT
def is_valid_draught(draught_candidate):
    """Sjekker om draught er en gyldig positiv verdi (f.eks. mellom 0.1 og 50.0)."""
    if draught_candidate is None:
        return False
    try:
        # Konverter til Decimal for nøyaktig sammenligning
        draught_val = Decimal(str(draught_candidate))
        # Antar at draught må være positivt og innenfor et rimelig område.
        # 0 kan være gyldig hvis det betyr "ikke satt" eller "på overflaten", avhengig av din definisjon.
        # Her setter vi en øvre grense på f.eks. 50 meter.
        if 0 <= draught_val <= Decimal('50.0'): # Tillater 0. Juster området etter behov.
            return True
    except InvalidOperation: # Håndterer hvis draught_candidate ikke kan konverteres til Decimal
        return False
    return False



def get_ship_type_description(ship_type_code):
    """Konverterer AIS skipstypekode til beskrivelse."""
    try:
        code = int(ship_type_code)
    except (ValueError, TypeError):
        return f"Ugyldig kode ({ship_type_code})"

    # Oppdatert liste basert på vanlige AIS standarder
    ship_types = {
        0: "Not Available / Unknown",
        # 1-19: Reserved
        20: "Wing In Ground (WIG), all ships of this type",
        21: "Wing In Ground (WIG), Hazardous category A",
        22: "Wing In Ground (WIG), Hazardous category B",
        23: "Wing In Ground (WIG), Hazardous category C",
        24: "Wing In Ground (WIG), Hazardous category D",
        25: "Wing In Ground (WIG), Reserved for future use",
        26: "Wing In Ground (WIG), Reserved for future use",
        27: "Wing In Ground (WIG), Reserved for future use",
        28: "Wing In Ground (WIG), Reserved for future use",
        29: "Wing In Ground (WIG), Reserved for future use",
        30: "Fishing",
        31: "Towing",
        32: "Towing: length exceeds 200m or breadth exceeds 25m",
        33: "Dredging or underwater ops",
        34: "Diving ops",
        35: "Military ops",
        36: "Sailing",
        37: "Pleasure Craft",
        38: "Reserved",
        39: "Reserved",
        40: "High speed craft (HSC), all ships of this type",
        41: "High speed craft (HSC), Hazardous category A",
        42: "High speed craft (HSC), Hazardous category B",
        43: "High speed craft (HSC), Hazardous category C",
        44: "High speed craft (HSC), Hazardous category D",
        45: "High speed craft (HSC), Reserved for future use",
        46: "High speed craft (HSC), Reserved for future use",
        47: "High speed craft (HSC), Reserved for future use",
        48: "High speed craft (HSC), Reserved for future use",
        49: "High speed craft (HSC), No additional information",
        50: "Pilot Vessel",
        51: "Search and Rescue vessel",
        52: "Tug",
        53: "Port Tender",
        54: "Anti-pollution equipment",
        55: "Law Enforcement",
        56: "Spare - Local Vessel",
        57: "Spare - Local Vessel",
        58: "Medical Transport",
        59: "Non-combatant ship according to RR Resolution No. 18",
        60: "Passenger, all ships of this type",
        61: "Passenger, Hazardous category A",
        62: "Passenger, Hazardous category B",
        63: "Passenger, Hazardous category C",
        64: "Passenger, Hazardous category D",
        65: "Passenger, Reserved for future use",
        66: "Passenger, Reserved for future use",
        67: "Passenger, Reserved for future use",
        68: "Passenger, Reserved for future use",
        69: "Passenger, No additional information",
        70: "Cargo, all ships of this type",
        71: "Cargo, Hazardous category A",
        72: "Cargo, Hazardous category B",
        73: "Cargo, Hazardous category C",
        74: "Cargo, Hazardous category D",
        75: "Cargo, Reserved for future use",
        76: "Cargo, Reserved for future use",
        77: "Cargo, Reserved for future use",
        78: "Cargo, Reserved for future use",
        79: "Cargo, No additional information",
        80: "Tanker, all ships of this type",
        81: "Tanker, Hazardous category A",
        82: "Tanker, Hazardous category B",
        83: "Tanker, Hazardous category C",
        84: "Tanker, Hazardous category D",
        85: "Tanker, Reserved for future use",
        86: "Tanker, Reserved for future use",
        87: "Tanker, Reserved for future use",
        88: "Tanker, Reserved for future use",
        89: "Tanker, No additional information",
        90: "Other Type, all ships of this type",
        91: "Other Type, Hazardous category A",
        92: "Other Type, Hazardous category B",
        93: "Other Type, Hazardous category C",
        94: "Other Type, Hazardous category D",
        95: "Other Type, Reserved for future use",
        96: "Other Type, Reserved for future use",
        97: "Other Type, Reserved for future use",
        98: "Other Type, Reserved for future use",
        99: "Other Type, no additional information"
    }
    return ship_types.get(code, f"Ukjent type ({code})")


def is_valid_mmsi(mmsi_str):
    """Sjekker om MMSI er gyldig (9 siffer, kun tall)."""
    return isinstance(mmsi_str, str) and mmsi_str.isdigit() and len(mmsi_str) == 9


def determine_ship_type_from_mmsi(mmsi):
    """Bestem skipstype basert på MMSI-prefiks (forenklet)."""
    if not is_valid_mmsi(mmsi):
        return "Ukjent"

    # Spesialtilfeller først (mer pålitelige)
    if mmsi.startswith("111"): return "SAR Aircraft"
    if mmsi.startswith("970"): return "AIS SART" # Search and Rescue Transponder
    if mmsi.startswith("972"): return "Man Overboard (MOB)"
    if mmsi.startswith("974"): return "EPIRB-AIS"
    if mmsi.startswith("99"): return "Navigation Aid" # AtoN
    if mmsi.startswith("00"): return "Base Station"
    if mmsi.startswith("0"): return "Group Call / Base Station" # Kan være base stasjon

    # Basert på første siffer (mindre presist, kun som fallback)
    first_digit = int(mmsi[0])
    if first_digit in [2, 3]: return "Standard Vessel (Check Code)" # Dekker de fleste land
    # Ingen flere pålitelige slutninger basert kun på første siffer

    return "Ukjent" # Fallback


def is_valid_callsign(candidate, ship_name, mmsi):
    """ Sjekker om callsign er gyldig. """
    if not candidate or not isinstance(candidate, str): return False
    candidate = candidate.strip().upper()

    if not (4 <= len(candidate) <= 7): return False
    if not re.fullmatch(r"[A-Z0-9]+", candidate): return False
    if not re.search(r"[A-Z]", candidate): return False # Må ha minst en bokstav
    if candidate in ["MMSI", "SEARCH", "NUMBER", "UKJENT", "UNKNOWN", str(mmsi)]: return False
    if ship_name and ship_name.strip() and ship_name.lower() != "ukjent":
        ship_name_words = set(ship_name.upper().split())
        if candidate in ship_name_words: return False
    return True


def is_valid_ship_name(candidate, mmsi):
    """ Validerer skipsnavn. """
    if not candidate or not isinstance(candidate, str): return False
    candidate = candidate.strip().upper()

    if candidate == str(mmsi): return False
    if candidate.isdigit(): return False
    if len(candidate) < 3: return False
    if not re.search(r"[A-Z]", candidate): return False
    if candidate in ["MMSI", "SEARCH", "NUMBER", "VESSEL", "SHIP", "UKJENT", "UNKNOWN", "NONE", "NULL"]: return False
    if str(mmsi) in candidate: return False
    # Unngå navn som bare er landskoder etc.
    if len(candidate) <= 4 and re.fullmatch(r"[A-Z]{2,4}", candidate): return False

    return True


# --- Data Extraction Functions (Regex) ---

def extract_ship_name(text, mmsi):
    """ Prøver å hente skipsnavn fra tekst. """
    if not text: return None
    patterns = [
        r"Vessel\s+([A-Z0-9\s\-']{3,30})\s+(?:is\s+a|MMSI|\(MMSI)",
        r"Ship\s+([A-Z0-9\s\-']{3,30})\s+(?:is\s+a|MMSI|\(MMSI)",
        r"Name:\s*([A-Z0-9\s\-']{3,30})\b",
        r"Vessel details for:\s*([A-Z0-9\s\-']{3,30})\b",
        r"current position of the\s+(?:(?:[A-Za-z]+\s+)*?)\s*'?([A-Z0-9\s\-']{3,30})'?\s+\(MMSI",
        r"Details for the ship\s+([A-Z0-9\s\-']{3,30})\s+\((?:IMO|MMSI)",
        # Fra tittelen (mindre presist, men kan fange opp)
        r"<title>.*? Vessel (.*?) - .*?</title>",
        r"<title>(.*?) \(MMSI: \d+\).*?</title>"
    ]
    for pattern in patterns:
        # Bruk re.IGNORECASE for å være uavhengig av store/små bokstaver i HTML/tekst
        match = re.search(pattern, text, re.IGNORECASE)
        if match:
            candidate = match.group(1).strip().upper()
            # Fjern potensielle rester av HTML e.l.
            candidate = re.sub(r'<.*?>', '', candidate).strip()
            if is_valid_ship_name(candidate, mmsi):
                return candidate
    return None


def extract_callsign(text, ship_name, mmsi):
    """ Prøver å hente callsign fra tekst. """
    if not text: return None
    patterns = [
        r"Call Sign[:\s]*([A-Z0-9]{4,7})\b",
        r"Callsign[:\s]*([A-Z0-9]{4,7})\b",
    ]
    for pattern in patterns:
        match = re.search(pattern, text, re.IGNORECASE)
        if match:
            candidate = match.group(1).strip().upper()
            if is_valid_callsign(candidate, ship_name, mmsi):
                return candidate
    return None


def extract_ship_type(text, mmsi):
    """ Prøver å hente skipstype fra tekst. """
    if not text: return None
    # Utvidet liste med flere vanlige typer
    valid_ship_types = {
        "Passenger Ship", "Cargo Ship", "Tanker", "High Speed Craft", "HSC",
        "Fishing Vessel", "Fishing", "Pleasure Craft", "Sailing Vessel", "Tug",
        "Pilot Vessel", "Search and Rescue", "SAR", "Law Enforcement",
        "Port Tender", "Anti-Pollution", "Medical Transport", "Military Ops",
        "Other Type", "Navigation Aid", "AtoN", "Reference Point", "Base Station",
        "SAR Aircraft", "Dredger", "Supply Vessel", "Offshore", "Yacht",
        "Wing In Ground", "WIG", "Diving Ops", "Underwater Ops", "Towing"
    }
    valid_lower = {t.lower() for t in valid_ship_types}

    # 1. Søk etter "is a [Type]", "type: [Type]", etc.
    patterns = [
        r"\b(?:is a|vessel is a|ship is a|type is|Type:)\s+([\w\s\-()]+)\b",
    ]
    for pattern in patterns:
        match = re.search(pattern, text, re.IGNORECASE)
        if match:
            candidate = match.group(1).strip().title()
            # Fjern ting som "(Hazard Cat...)"
            candidate = re.sub(r'\s*\(.*?\)', '', candidate).strip()
            if candidate.lower() in valid_lower:
                if candidate.lower() != str(mmsi).lower() and candidate.lower() != "mmsi":
                    # Map vanlige forkortelser til fullt navn
                    if candidate.upper() == 'HSC': candidate = 'High Speed Craft'
                    if candidate.upper() == 'SAR': candidate = 'Search and Rescue'
                    if candidate.upper() == 'WIG': candidate = 'Wing In Ground'
                    if candidate.upper() == 'ATON': candidate = 'Navigation Aid'
                    return candidate

    # 2. Søk etter kjente typer som ord i teksten
    # Omdøpt variabel for å unngå navnekonflikt med den globale variabelen `ship_type`
    sorted_types = sorted(list(valid_ship_types), key=len, reverse=True)
    for current_ship_type_candidate in sorted_types:
        # Bruk re.escape for å håndtere spesielle tegn i type-navn
        # Bruk \b for å sikre hele ord
        if re.search(r'\b' + re.escape(current_ship_type_candidate) + r'\b', text, re.IGNORECASE):
            context_match = re.search(r'(?:[\s\(\.\,:/]|^)' + re.escape(current_ship_type_candidate) + r'(?:[\s\)\.\,:/]|$)', text, re.IGNORECASE)
            if context_match:
                if current_ship_type_candidate.lower() != str(mmsi).lower() and current_ship_type_candidate.lower() != "mmsi":
                    # Map forkortelser
                    if current_ship_type_candidate.upper() == 'HSC': current_ship_type_candidate = 'High Speed Craft'
                    if current_ship_type_candidate.upper() == 'SAR': current_ship_type_candidate = 'Search and Rescue'
                    if current_ship_type_candidate.upper() == 'WIG': current_ship_type_candidate = 'Wing In Ground'
                    if current_ship_type_candidate.upper() == 'ATON': current_ship_type_candidate = 'Navigation Aid'
                    return current_ship_type_candidate.title() # Returner med store forbokstaver
    return None


def extract_image_url(text):
    """ Henter bilde-URL fra tekst. Prøver å unngå små/generiske bilder. """
    if not text: return None
    # Søk etter bilde-URLer, prioriter de som ser ut som hovedbilder
    patterns = [
        # Direkte bilde-linker
        r'(https?://\S+\.(?:jpg|jpeg|png|gif))\b',
        # MarineTraffic
        r'(https?://photos\.marinetraffic\.com/ais/showphoto\.aspx\?photoid=\d+)',
        # VesselFinder o.l.
        r'(https?://(?:www\.)?vesselfinder\.com/vessel-photos/\S+\.(?:jpg|jpeg|png|gif))',
        r'src="(https?://(?:cdn|static)\.\S+?/photos?/\S+\.(?:jpg|jpeg|png|gif))"',
        # Generell <img> tag src
        r'<img[^>]+src="([^">]+\.(?:jpg|jpeg|png|gif))"',
    ]
    for pattern in patterns:
        matches = re.finditer(pattern, text, re.IGNORECASE)
        for match in matches:
            url = match.group(1).strip()
            # Fjern eventuelle url parametre etter filendelsen
            url = re.sub(r'(\.(?:jpg|jpeg|png|gif)).*', r'\1', url, flags=re.IGNORECASE)

            # Filtrer bort kjente uønskede mønstre
            ignore_patterns = ["logo", "icon", "avatar", "thumb", "sprite", "flag", "banner", "button",
                               "captcha", "spacer", "loading", "ogImg", "favicon", "small", "tiny",
                               "marker", "pixel", "/ads/", "/banners/", "/widgets/", "_s.", "_t.", "_sq.",
                               "16x16", "32x32", "64x64", "100x100"]
            if not any(ignore in url.lower() for ignore in ignore_patterns):
                 # Sjekk om URLen er plausibel
                if url.startswith('http') and url.count('.') >= 1 and url.count('/') >= 2:
                     print(f"      --> Fant potensiell bilde-URL: {url}")
                     return url # Returner første gode treff

    return None


def download_image(image_url, mmsi):
    """ Laster ned bildet hvis URL er gyldig. Returnerer filnavn eller None. """
    # Bruk global IMAGE_FOLDER
    global IMAGE_FOLDER
    try:
        img_folder_path = IMAGE_FOLDER
    except NameError:
        print("⚠️ IMAGE_FOLDER er ikke definert globalt!")
        return None # Kan ikke laste ned uten mappe

    if not image_url:
        return None

    # Enkel sjekk av URL før forsøk
    if not image_url.startswith('http'):
        print(f"⚠️ Ignorerer ugyldig bilde-URL (mangler http/https): {image_url}")
        return None

    try:
        print(f"   ⬇️ Prøver å laste ned bilde for {mmsi} fra: {image_url}")
        # Legg til User-Agent for å unngå blokkering
        headers = {'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'}
        response = requests.get(image_url, stream=True, timeout=20, headers=headers, allow_redirects=True)
        response.raise_for_status()

        content_type = response.headers.get('Content-Type', '').lower()
        if not content_type.startswith('image/'):
            print(f"   ⚠️ URL returnerte ikke en bildetype ({content_type}), hopper over.")
            return None

        # Bestem filendelse (default til .jpg)
        extension = ".jpg"
        if 'jpeg' in content_type: extension = ".jpg" # Bruk jpg for jpeg
        elif 'png' in content_type: extension = ".png"
        elif 'gif' in content_type: extension = ".gif"

        image_filename = f"{mmsi}{extension}"
        image_path = os.path.join(img_folder_path, image_filename)

        # Sjekk om filen allerede finnes, ikke last ned på nytt
        if os.path.exists(image_path):
            print(f"   ℹ️ Bilde {image_filename} finnes allerede, hopper over nedlasting.")
            return image_filename # Returner eksisterende filnavn

        with open(image_path, "wb") as file:
            for chunk in response.iter_content(1024 * 8): # Litt større chunk
                file.write(chunk)
        print(f"   ✅ Bilde lagret: {image_path}")
        return image_filename
    except requests.exceptions.Timeout:
        print(f"   ⚠️ Timeout ved nedlasting av bilde {image_url} for MMSI {mmsi}.")
    except requests.exceptions.RequestException as e:
        print(f"   ⚠️ Feil ved nedlasting av bilde {image_url} for MMSI {mmsi}: {e}")
    except Exception as e:
        print(f"   ⚠️ Uventet feil under bildenedlasting for MMSI {mmsi}: {e}")
    return None


# --- Søkefunksjoner ---

def search_aishub_api(mmsi):
    """ 
    Slår opp skipets data i AISHub API. Returnerer dictionary eller None.
    Inkluderer heuristikk for å tolke DRAUGHT-enhet.
    """
    global AISHUB_USERNAME, AISHUB_API_URL
    try:
        username = AISHUB_USERNAME
        api_url = AISHUB_API_URL
    except NameError:
        print("⚠️ AISHUB_USERNAME eller AISHUB_API_URL er ikke definert globalt!")
        return None

    print(f"   🌐 Henter data fra AISHub API for MMSI {mmsi}...")
    params = {
        "username": username, "format": "1", "output": "json",
        "compress": 0, "mmsi": mmsi
    }
    response = None
    try:
        response = requests.get(api_url, params=params, timeout=25)
        response.raise_for_status()

        if not response.text.strip():
            print(f"   ⚠️ AISHub API returnerte tom respons for MMSI {mmsi}.")
            return None
        try:
            response_data = response.json()
        except json.JSONDecodeError as json_err:
            print(f"   ⚠️ Kunne ikke dekode JSON fra AISHub for MMSI {mmsi}: {json_err}")
            print(f"      Respons (start): {response.text[:200]}")
            return None

        if isinstance(response_data, list) and len(response_data) > 0:
            if isinstance(response_data[0], dict) and response_data[0].get('ERROR', False):
                error_msg = response_data[0].get('ERROR_MESSAGE', 'Ukjent feil i API-respons')
                print(f"   ⚠️ AISHub API rapporterte feil: {error_msg}")
                return None
            if len(response_data) < 2 or not isinstance(response_data[1], list) or not response_data[1]:
                print(f"   ⚠️ AISHub API returnerte gyldig JSON, men manglet data for MMSI {mmsi}.")
                return None
        else:
            print(f"   ⚠️ Uventet JSON-struktur fra AISHub API for MMSI {mmsi}.")
            return None

        ship_info_data = response_data[1][0]
        api_name = ship_info_data.get("NAME")
        api_callsign = ship_info_data.get("CALLSIGN")
        api_type_code = ship_info_data.get("TYPE")
        api_ship_type = get_ship_type_description(api_type_code) if api_type_code is not None else None
        api_imo_val = ship_info_data.get("IMO")
        api_draught_value = ship_info_data.get("DRAUGHT") # Hent råverdien for DRAUGHT

        name = str(api_name).strip() if api_name and is_valid_ship_name(str(api_name), mmsi) else None
        callsign = str(api_callsign).strip() if api_callsign and is_valid_callsign(str(api_callsign), name or "", mmsi) else None
        ship_type_result = api_ship_type if api_ship_type and not api_ship_type.startswith("Ukjent type") else None
        
        imo_result = None
        if api_imo_val is not None and is_valid_imo(str(api_imo_val)): # Sikre streng til is_valid_imo
            imo_result = int(api_imo_val)
        elif api_imo_val is not None:
            print(f"   ⚠️ AISHub IMO '{api_imo_val}' ansett som ugyldig.")

        draught_result = None
        if api_draught_value is not None:
            try:
                val_str = str(api_draught_value)
                val_decimal = Decimal(val_str) # Konverter til Decimal én gang

                # Heuristikk for å bestemme enhet:
                # Hvis verdien inneholder et desimalpunkt (f.eks. "6.3"), anta at den allerede er i meter.
                # Hvis verdien er et heltall (f.eks. "63"), anta at den er i desimeter og del på 10.
                if '.' in val_str: 
                    draught_val_meters = val_decimal # Anta at verdien allerede er i meter
                    print(f"   ℹ️ AISHub DRAUGHT '{api_draught_value}' tolket som meter direkte (pga. desimalpunkt).")
                else:
                    draught_val_meters = val_decimal / Decimal('10.0') # Anta desimeter, konverter til meter
                    print(f"   ℹ️ AISHub DRAUGHT '{api_draught_value}' tolket som desimeter, konverterer til {draught_val_meters:.1f}m.")
                
                if is_valid_draught(draught_val_meters): # Valider den (potensielt) konverterte verdien
                    draught_result = draught_val_meters
                else:
                    # Forbedret loggmelding for ugyldig draught etter heuristikk
                    print(f"   ⚠️ AISHub DRAUGHT {api_draught_value} (tolket/konvertert til {draught_val_meters:.2f}m) ansett som ugyldig av is_valid_draught.")
            except (TypeError, ValueError, InvalidOperation) as e:
                print(f"   ⚠️ Kunne ikke konvertere AISHub DRAUGHT '{api_draught_value}' til gyldig tall: {e}")

        result_data = {
            "ship_name": name, 
            "callsign": callsign, 
            "ship_type": ship_type_result, 
            "imo": imo_result,
            "draught": draught_result,
            "image_url": None 
        }
        
        found_data_log = {k: v for k, v in result_data.items() if v is not None}
        if draught_result is not None: # Spesialformatering for draught i logg
            found_data_log["draught"] = f"{draught_result:.2f}m" # Vis med 2 desimaler i loggen

        print(f"   ✅ Data hentet fra AISHub API: {found_data_log}")
        return result_data

    except requests.exceptions.Timeout:
        print(f"   ⚠️ Timeout ved kall til AISHub API for MMSI {mmsi}.")
    except requests.exceptions.RequestException as e:
        print(f"   ⚠️ Nettverksfeil ved kall til AISHub API for MMSI {mmsi}: {e}")
        if response is not None: print(f"      Status: {response.status_code}, Tekst (start): {response.text[:200]}")
    except (IndexError, KeyError, TypeError) as e:
        print(f"   ⚠️ Feil ved tolking av AISHub JSON-data for MMSI {mmsi}: {e}")
        if 'response_data' in locals() and response_data: print(f"      Mottatt data (start): {str(response_data)[:500]}")
    except Exception as e:
        print(f"   ⚠️ Uventet feil i search_aishub_api for MMSI {mmsi}: {e}")
    return None



# Definer headers, se ut som nettleser
REQUEST_HEADERS = {
    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/108.0.0.0 Safari/537.36',
    'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9',
    'Accept-Language': 'nb-NO,nb;q=0.9,no;q=0.8,en-US;q=0.7,en;q=0.6', # Foretrekk norsk
    'Connection': 'keep-alive',
    'DNT': '1' # Do Not Track
}

def search_kystverket(mmsi):
    """Henter statisk skipsdata fra Kystverkets offentlige API."""
    # Kystverkets API forventer MMSI i URL-en
    target_url = f"https://nais.kystverket.no/api/v1/vessel/{mmsi}"
    print(f"  🚢 Henter data fra Kystverket (NAIS) for MMSI {mmsi}...")
    
    # Bruk de samme headers som før for å se ut som en nettleser
    headers = {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/108.0.0.0 Safari/537.36',
        'Accept': 'application/json'
    }

    try:
        response = requests.get(target_url, headers=headers, timeout=15)
        response.raise_for_status() # Sjekker for feil som 404 (Not Found)
        
        data = response.json()
        
        # Sjekk om responsen inneholder data og ikke er en feilmelding
        if not data or not isinstance(data, dict):
            print(f"  ⚠️ Kystverket ga tom eller ugyldig respons for MMSI {mmsi}.")
            return None

        # Hent ut de relevante datafeltene
        ship_name = data.get('name')
        callsign = data.get('callSign')
        imo = data.get('imo')
        ship_type_code = data.get('shipType') # Kystverket bruker 'shipType' for koden

        # Valider og rens dataene før retur
        ship_name_clean = ship_name.strip() if ship_name and is_valid_ship_name(ship_name, mmsi) else None
        callsign_clean = callsign.strip() if callsign and is_valid_callsign(callsign, ship_name, mmsi) else None
        imo_clean = int(imo) if imo and is_valid_imo(imo) else None
        ship_type_desc = get_ship_type_description(ship_type_code) if ship_type_code is not None else None
        
        result = {
            "ship_name": ship_name_clean,
            "callsign": callsign_clean,
            "ship_type": ship_type_desc,
            "imo": imo_clean,
            "draught": None, # Denne informasjonen er ikke i dette API-kallet
            "image_url": None # Heller ikke bilde-URL
        }
        
        # Logg kun de feltene vi faktisk fant
        found_data_log = {k: v for k, v in result.items() if v is not None}
        if not found_data_log:
             print(f"  ℹ️ Kystverket hadde ingen ny/gyldig info for MMSI {mmsi}.")
             return None
             
        print(f"  ✅ Data hentet fra Kystverket: {found_data_log}")
        return result

    except requests.exceptions.HTTPError as e:
        if e.response.status_code == 404:
            print(f"  ℹ️ Kystverket fant ikke MMSI {mmsi} (404 Not Found).")
        else:
            print(f"  ❌ HTTP-feil fra Kystverket for MMSI {mmsi}: {e}")
        return None
    except requests.exceptions.RequestException as e:
        print(f"  ❌ Nettverksfeil mot Kystverket for MMSI {mmsi}: {e}")
        return None
    except Exception as e:
        print(f"  ❌ Uventet feil i search_kystverket for MMSI {mmsi}: {e}")
        return None




def fetch_ship_details_trackipi(mmsi):
    """
    Fallback-funksjon: Henter skipsdetaljer fra trackipi.com.

    Bruker web scraping. Trackipi's vilkår ser ut til å tillate bruk av dataen,
    men funksjonen kan slutte å virke hvis nettstedets HTML endres.
    Scraping bør fortsatt utføres på en hensynsfull måte.

    Args:
        mmsi (str or int): Skipets MMSI-nummer.

    Returns:
        dict: Ordbok med funnet skipsinfo (f.eks. 'name', 'imo', 'callsign', 'type')
              eller None ved feil, eller tom ordbok {} hvis siden hentes OK,
              men ingen spesifikk data finnes med nåværende regler.
    """
    if not mmsi:
        print("❌ MMSI mangler for Trackipi fallback.")
        return None

    # Merk URL-strukturen her er /vessel/mmsi-<MMSI>
    target_url = f"https://www.trackipi.com/vessel/mmsi-{mmsi}"
    # Bruk f-string for enkel formatering
    print(f"ℹ️ Forsøker fallback [?] til Trackipi for MMSI {mmsi}: {target_url}") # Angi fallback-nummer selv

    try:
        with requests.Session() as session:
            session.headers.update(REQUEST_HEADERS)
            response = session.get(target_url, timeout=20)

        response.raise_for_status() # Sjekker for HTTP-feil

        try:
            soup = BeautifulSoup(response.text, 'lxml')
        except: # noqa
            soup = BeautifulSoup(response.text, 'html.parser')

        # --- START PARSING AV HTML ---
        # VIKTIG: Erstatt kommentarer/eksempler med faktiske selektorer
        # basert på inspeksjon av HTML-kilden for en Trackipi-side.

        ship_details = {}

        # Eksempel 1: Finne skipsnavn (ofte i en <h1> tag)
        # TODO: Inspiser HTML for å finne tag/klasse/id for skipsnavn
        name_element = soup.find('h1', class_='vessel-name') # <-- ERSTATT MED RIKTIG SELECTOR
        if name_element:
            ship_details['name'] = name_element.get_text(strip=True)

        # Eksempel 2: Finne data fra en tabell eller liste (vanlig for detaljer)
        # Anta at data finnes i en tabell med <tr> rader, hvor første <td> er label og andre er verdi
        # TODO: Finn riktig tabell/liste-element
        details_table = soup.find('table', class_='vessel-details-table') # <-- ERSTATT MED RIKTIG SELECTOR
        if details_table:
            rows = details_table.find_all('tr')
            for row in rows:
                cells = row.find_all(['th', 'td']) # Finn både label (th?) og verdi (td)
                if len(cells) == 2:
                    label = cells[0].get_text(strip=True).lower().replace(':', '').replace(' ', '_')
                    value = cells[1].get_text(strip=True)

                    # Map kjente labels til ordbok-nøkler
                    if 'imo' in label:
                        ship_details['imo'] = value
                    elif 'call_sign' in label or 'callsign' in label:
                        ship_details['callsign'] = value
                    elif 'flag' in label:
                        ship_details['flag'] = value
                    # Omdøpt variabel for å unngå konflikt
                    elif 'type' in label or 'vessel_type' in label:
                        ship_details['type'] = value # Kan trenge videre behandling/mapping
                    elif 'length' in label and 'beam' in label: # Ofte samlet
                        ship_details['dimensions'] = value
                    elif 'gross_tonnage' in label or label == 'gt':
                         ship_details['gt'] = value
                    elif 'deadweight' in label or label == 'dwt':
                         ship_details['dwt'] = value
                    elif 'year_built' in label or 'built' in label:
                         ship_details['year_built'] = value
                    # Legg til flere etter behov...

        # Eksempel 3: Finne posisjonsdata (kan være i egne div-er eller en kart-datadel)
        # TODO: Inspiser HTML for lat/lon, speed, course, status
        # lat_element = soup.find('span', class_='latitude-value') # <-- ERSTATT
        # lon_element = soup.find('span', class_='longitude-value') # <-- ERSTATT
        # if lat_element and lon_element:
        #     ship_details['latitude'] = lat_element.get_text(strip=True)
        #     ship_details['longitude'] = lon_element.get_text(strip=True)
        #
        # speed_element = soup.find('div', id='speed-info') # <-- ERSTATT
        # if speed_element:
        #     ship_details['speed'] = speed_element.get_text(strip=True)


        # --- SLUTT PÅ PARSING ---

        if ship_details:
            print(f"✅ Fant skipsdetaljer fra Trackipi for MMSI {mmsi}")
            # Legg til MMSI i resultatene for enkelhets skyld
            ship_details['mmsi'] = str(mmsi)
            ship_details['source'] = 'trackipi.com' # Identifiser kilden
            return ship_details
        else:
            print(f"⚠️ Hentet side fra Trackipi for {mmsi}, men fant ingen spesifikk info med nåværende regler.")
            return {} # Returner tom dict

    except requests.exceptions.HTTPError as e:
        if e.response.status_code == 404:
            print(f"ℹ️ Trackipi returnerte 404 (Not Found) for MMSI {mmsi}.")
        else:
            print(f"❌ HTTP-feil under henting fra Trackipi for MMSI {mmsi}: {e}")
        return None
    except requests.exceptions.RequestException as e:
        print(f"❌ Nettverksfeil under henting fra Trackipi for MMSI {mmsi}: {e}")
        return None
    except Exception as e:
        # Andre uventede feil (f.eks. under parsing med BeautifulSoup)
        print(f"❌ Uventet feil under behandling av Trackipi for MMSI {mmsi}: {e}")
        return None


def fetch_ship_journey_harbourmaps(mmsi):
    """
    Fallback-funksjon nr. 4: Henter reiseinformasjon for et skip fra harbourmaps.com.

    Bruker web scraping. Kan slutte å virke hvis nettstedets HTML endres.
    Bruk hensynsfullt for å unngå overbelastning av serveren.

    Args:
        mmsi (str or int): Skipets MMSI-nummer.

    Returns:
        dict: Ordbok med funnet reiseinfo (f.eks. 'destination', 'eta', 'port_calls')
              eller None ved feil, eller tom ordbok {} hvis siden hentes OK,
              men ingen spesifikk data finnes med nåværende regler.
    """

    if not mmsi:
        print("❌ MMSI mangler for Harbourmaps fallback.")
        return None

    target_url = f"https://www.harbourmaps.com/ship/journey/q?&mmsi={mmsi}"
    print(f"ℹ️ Forsøker fallback [4] til Harbourmaps for MMSI {mmsi}: {target_url}")

    try:
        # Bruk økt for potensiell gjenbruk av tilkobling
        with requests.Session() as session:
            session.headers.update(REQUEST_HEADERS)
            response = session.get(target_url, timeout=20) # Økt timeout til 20 sekunder

        response.raise_for_status()  # Sjekker for HTTP-feil (f.eks. 404 Not Found, 500 Server Error)

        # Bruk lxml parser hvis installert (ofte raskere), ellers html.parser
        try:
            soup = BeautifulSoup(response.text, 'lxml')
        except: # noqa
            soup = BeautifulSoup(response.text, 'html.parser')

        # --- START PARSING AV HTML ---
        # VIKTIG: Erstatt kommentarene og eksempel-selektorene nedenfor
        # med de faktiske selektorene du finner ved å inspisere HTML-kilden.

        journey_data = {}

        # Eksempel 1: Finne destinasjon
        # TODO: Inspiser HTML og finn riktig element/klasse/id for destinasjon
        # Anta at destinasjon er i en <p> tag med klassen 'ship-destination'
        dest_element = soup.find('p', class_='ship-destination') # <-- ERSTATT MED RIKTIG SELECTOR
        if dest_element:
            journey_data['destination'] = dest_element.get_text(strip=True)

        # Eksempel 2: Finne ETA
        # TODO: Inspiser HTML og finn riktig element/klasse/id for ETA
        # Anta at ETA er i en <td> tag etter en <th> tag som inneholder "ETA"
        eta_label = soup.find('th', string=lambda t: t and 'ETA' in t)
        if eta_label and eta_label.find_next_sibling('td'):
             journey_data['eta'] = eta_label.find_next_sibling('td').get_text(strip=True)
        # Alternativt, finn en spesifikk ID eller klasse hvis den finnes.
        # eta_element = soup.find(id='eta_field') # <-- ERSTATT MED RIKTIG SELECTOR
        # if eta_element:
        #     journey_data['eta'] = eta_element.get_text(strip=True)

        # Eksempel 3: Finne tabell med havneanløp ('Port Calls')
        # TODO: Inspiser HTML og finn riktig tabell (f.eks. basert på en <caption> eller id/klasse)
        port_calls_table = soup.find('table', id='port-calls-history') # <-- ERSTATT MED RIKTIG SELECTOR
        if port_calls_table:
            port_calls = []
            # TODO: Loop gjennom radene (<tr>) i tabellens body (<tbody>)
            #       og trekk ut data fra cellene (<td>). Tilpass til kolonnene.
            table_body = port_calls_table.find('tbody')
            if table_body:
                for row in table_body.find_all('tr'):
                    cells = row.find_all('td')
                    # Anta kolonner: Havn, Ankomst, Avgang
                    if len(cells) >= 3:
                        call = {
                            'port': cells[0].get_text(strip=True),
                            'arrival': cells[1].get_text(strip=True),
                            'departure': cells[2].get_text(strip=True)
                        }
                        port_calls.append(call)
                    elif len(cells) == 2: # Kanskje bare havn og ankomst?
                         call = {
                            'port': cells[0].get_text(strip=True),
                            'arrival': cells[1].get_text(strip=True),
                            'departure': None # Eller ""
                        }
                         port_calls.append(call)

            if port_calls:
                journey_data['port_calls'] = port_calls

        # Eksempel 4: Finne "Last Port"
        # TODO: Inspiser HTML og finn riktig element/klasse/id for siste havn
        # last_port_el = soup.find('div', {'data-label': 'Last Port'}) # Eksempel på annen type selector
        # if last_port_el:
        #    journey_data['last_port'] = last_port_el.get_text(strip=True)


        # --- SLUTT PÅ PARSING ---

        if journey_data:
            print(f"✅ Fant reiseinfo fra Harbourmaps for MMSI {mmsi}")
            return journey_data
        else:
            print(f"⚠️ Hentet side fra Harbourmaps for {mmsi}, men fant ingen spesifikk reiseinfo med nåværende regler.")
            # Returner tom dict for å signalisere at henting var ok, men ingen data funnet
            return {}

    except requests.exceptions.HTTPError as e:
        # Spesifikk håndtering for HTTP-feil (f.eks. 404 - skip finnes ikke)
        if e.response.status_code == 404:
            print(f"ℹ️ Harbourmaps returnerte 404 (Not Found) for MMSI {mmsi}. Skipet finnes trolig ikke der.")
        else:
            print(f"❌ HTTP-feil under henting fra Harbourmaps for MMSI {mmsi}: {e}")
        return None
    except requests.exceptions.RequestException as e:
        # Andre nettverksfeil (timeout, DNS-feil etc.)
        print(f"❌ Nettverksfeil under henting fra Harbourmaps for MMSI {mmsi}: {e}")
        return None
    except Exception as e:
        # Andre uventede feil (f.eks. under parsing med BeautifulSoup)
        print(f"❌ Uventet feil under behandling av Harbourmaps for MMSI {mmsi}: {e}")
        return None



def search_brave(mmsi, existing_ship_name, existing_callsign):
    """ Søker på Brave Search. Returnerer dictionary eller None. """
    # Importer BeautifulSoup inne i funksjonen for å unngå feil hvis den mangler
    global BeautifulSoup # Bruk global (ekte eller dummy)

    print(f"   🔍 Søker på Brave for MMSI {mmsi}...")
    query_parts = [f"MMSI {mmsi}"]
    if existing_ship_name and existing_ship_name.lower() != "ukjent": query_parts.append(f'"{existing_ship_name}"')
    elif existing_callsign and existing_callsign.lower() != "ukjent": query_parts.append(existing_callsign)
    query_parts.append("ship details vessel information")
    query = " ".join(query_parts)
    url = f"https://search.brave.com/search?q={requests.utils.quote(query)}&source=web"
    print(f"      URL: {url}")

    headers = {'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36', 'Accept-Language': 'en-US,en;q=0.9'}

    try:
        response = requests.get(url, headers=headers, timeout=25) # Økt timeout
        response.raise_for_status()

        text_content = response.text # Behold rå HTML for bilde-URL søk
        plain_text = ""
        if BeautifulSoup.__name__ != 'DummySoup': # Kjør kun hvis ekte import var vellykket
             try:
                 soup = BeautifulSoup(response.content, 'html.parser')
                 plain_text = soup.get_text(separator=" ", strip=True)
             except Exception as bs_err:
                 print(f"   ⚠️ Feil under parsing av Brave HTML med BeautifulSoup: {bs_err}")
                 plain_text = text_content # Fallback til råtekst hvis parsing feiler
        else:
             plain_text = text_content # Bruk råtekst hvis BS4 ikke er tilgjengelig

        # Trekk ut data
        name = extract_ship_name(plain_text, mmsi)
        callsign = extract_callsign(plain_text, name or existing_ship_name, mmsi)
        # Omdøpt variabel for å unngå navnekonflikt
        ship_type_result = extract_ship_type(plain_text, mmsi)
        # Omdøpt variabel for å unngå navnekonflikt
        image_url_result = extract_image_url(text_content) # Søk i rå HTML for bilder

        result = {"ship_name": name, "callsign": callsign, "ship_type": ship_type_result, "image_url": image_url_result}
        print(f"   ✅ Resultat fra Brave: { {k: v for k, v in result.items() if v} }")
        return result

    except requests.exceptions.Timeout:
        print(f"   ⚠️ Timeout under Brave-søk for MMSI {mmsi}.")
        return None
    except requests.exceptions.RequestException as e:
        print(f"   ⚠️ Feil under Brave-søk for MMSI {mmsi}: {e}")
        return None
    except Exception as e:
        print(f"   ⚠️ Uventet feil i search_brave for MMSI {mmsi}: {e}")
        return None


def search_duckduckgo(mmsi, existing_ship_name, existing_callsign):
    """ Søker på DuckDuckGo via ddgs-biblioteket. Returnerer dictionary eller None. """
    print(f"   🦆 Søker på DuckDuckGo (DDGS) for MMSI {mmsi}...")
    query_parts = [f"MMSI {mmsi}"]
    if existing_ship_name and existing_ship_name.lower() != "ukjent": query_parts.append(f'"{existing_ship_name}"')
    elif existing_callsign and existing_callsign.lower() != "ukjent": query_parts.append(existing_callsign)
    query_parts.extend(["vessel information", "ship data", "call sign", "ship type"])
    query = " ".join(query_parts)
    print(f"      Query: {query}")

    try:
        # Bruk DDGS() direkte
        results = DDGS(headers={'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'}, timeout=25).text(query, max_results=5)
        # Konverter generator til liste
        search_results = list(results)

        if not search_results:
            print(f"   ⚠️ Fant ingen DDGS-resultater for {mmsi}.")
            return None

        combined_text = " ".join(result.get("body", "") for result in search_results if result.get("body"))
        # Legg også til titler for bedre kontekst
        combined_titles = " ".join(result.get("title", "") for result in search_results if result.get("title"))
        full_text = combined_titles + " " + combined_text

        name = extract_ship_name(full_text, mmsi)
        callsign = extract_callsign(full_text, name or existing_ship_name, mmsi)
        # Omdøpt variabel for å unngå navnekonflikt
        ship_type_result = extract_ship_type(full_text, mmsi)
        # Omdøpt variabel for å unngå navnekonflikt
        image_url_result = extract_image_url(full_text) # Prøv å finne URL i tekst

        result = {"ship_name": name, "callsign": callsign, "ship_type": ship_type_result, "image_url": image_url_result}
        print(f"   ✅ Resultat fra DDGS: { {k: v for k, v in result.items() if v} }")
        return result

    except Exception as e:
        print(f"   ⚠️ Feil under DuckDuckGo (DDGS) søk for MMSI {mmsi}: {e}")
        return None

# --- MySQL Database Funksjoner (Oppdatert) ---

def get_db_connection():
    """Oppretter og returnerer en MySQL databaseforbindelse."""
    conn = None
    try:
        conn = mysql.connector.connect(
            host=DB_HOST,
            user=DB_USER,
            password=DB_PASSWORD,
            database=DB_NAME
        )
        # print("Databaseforbindelse opprettet.") # Kan fjernes når det virker
    except MySQLError as e:
        print(f"🚨 Feil ved tilkobling til MySQL database '{DB_NAME}': {e}")
        # Vurder å avslutte hvis kritisk
        sys.exit(f"Kritisk feil: Kan ikke koble til databasen. Feil: {e}")
    return conn

def initialize_database():
    """
    Verifiserer at tilkobling til den konfigurerte MySQL-databasen ({DB_NAME})
    kan opprettes med gitte innstillinger.
    Scriptet vil avslutte (via get_db_connection) hvis tilkoblingen feiler.
    Denne funksjonen gjør ingen endringer i databasen eller tabeller.
    Forutsetter at databasen og tabellen ({SHIP_TABLE}) allerede eksisterer.
    """
    print(f"ℹ️ Verifiserer database-tilkobling ({DB_USER}@{DB_HOST}/{DB_NAME})...")
    conn = None  # Initialiser til None
    try:
        # get_db_connection() forsøker tilkobling.
        # Den vil kalle sys.exit() internt hvis tilkoblingen feiler.
        conn = get_db_connection()

        if conn and conn.is_connected():
            # Hvis vi når hit, var tilkoblingen vellykket.
            print(f"✅ Tilkobling til '{DB_NAME}' verifisert.")
        else:
            # Dette punktet bør teoretisk ikke nås hvis get_db_connection
            # enten returnerer en gyldig tilkobling eller avslutter scriptet.
            # Inkludert for sikkerhets skyld.
            print(f"🚨 Kunne ikke verifisere tilkobling (uventet tilstand etter forsøk).")
            sys.exit(1) # Avslutt hvis vi er i en uventet tilstand

    except Exception as e:
        # Fanger eventuelle andre uventede feil under selve verifiseringen
        print(f"🚨 Uventet feil under verifisering av database-tilkobling: {e}")
        sys.exit(1)

    finally:
        # Sørg for at test-tilkoblingen lukkes hvis den ble opprettet
        if conn and conn.is_connected():
            try:
                conn.close()
                # print("Databaseforbindelse (verifisering) lukket.") # Kan fjernes
            except MySQLError as close_err:
                # Ikke kritisk hvis lukking feiler, men greit å logge
                print(f"⚠️ Feil under lukking av test-tilkobling: {close_err}")

def get_ships_to_process():
    """
    Henter rader fra MySQL 'ships'-tabellen hvor KRITISK informasjon
    (navn, type, callsign (hvis forventet), ELLER IMO) ser ut til å mangle
    (er NULL eller tom streng for tekstfelt, NULL for imo)
    OG
    som ikke har blitt oppdatert de siste 7 dagene (eller aldri).
    Returnerer MAKSIMALT 20 rader per kall.
    """
    ships = []
    conn = get_db_connection()
    if not conn:
        return ships # Tom liste ved tilkoblingsfeil

    cursor = None
    max_ships_to_fetch = 40 # Definerer maks antall her
    try:
        cursor = conn.cursor(dictionary=True)

        # Definer typer som IKKE trenger callsign (uendret)
        types_without_callsign_tuple = (
            'Navigation Aid', 'Reference Point', 'Base Station',
            'SAR Aircraft', 'AIS SART', 'Man Overboard (MOB)', 'EPIRB-AIS'
        )
        # Sikrer at strenger har anførselstegn for SQL IN-klausulen
        sql_in_clause_str = ", ".join([f"'{t}'" for t in types_without_callsign_tuple])

        # Bygg WHERE-betingelsen: Mangler data OG ikke oppdatert nylig
        where_clause = f"""
        WHERE
        ( -- Start: Kriterier for manglende data --
            (ship_name IS NULL OR ship_name = '')
            OR (ship_type IS NULL OR ship_type = '')
            OR (
                (callsign IS NULL OR callsign = '')
                AND
                (ship_type IS NULL OR ship_type NOT IN ({sql_in_clause_str}))
               )
            OR (imo IS NULL)
        ) -- Slutt: Kriterier for manglende data --
        AND
        ( -- Start: Kriterier for tid siden sist oppdatering --
            last_updated IS NULL OR last_updated < NOW() - INTERVAL 7 DAY
        ) -- Slutt: Kriterier for tid siden sist oppdatering --
        """

        # Hele spørringen med LIMIT lagt til
        query = f"""
            SELECT mmsi, callsign, ship_name, ship_type, image_url, imo, draught, last_updated
            FROM `{SHIP_TABLE}`
            {where_clause}
            LIMIT {max_ships_to_fetch} -- <--- BEGRENSNING LAGT TIL HER
        """

        print(f"ℹ️ Henter opptil {max_ships_to_fetch} skip som mangler data (inkl. IMO) og ikke er oppdatert siste 7 dager fra '{DB_NAME}.{SHIP_TABLE}'...")
        # print(f"DEBUG SQL: {query}") # For feilsøking

        cursor.execute(query)
        ships = cursor.fetchall()
        print(f"ℹ️ Fant {len(ships)} rader (maks {max_ships_to_fetch}) som trenger sjekk/oppdatering basert på kriteriene.")

    except MySQLError as e:
        print(f"🚨 Databasefeil ved henting av skip som trenger oppdatering fra '{SHIP_TABLE}': {e}")
    finally:
        # Rydd opp
        if cursor:
            cursor.close()
        if conn and conn.is_connected():
            conn.close()
    return ships


def update_ship_data(mmsi, update_data):
    """
    Oppdaterer spesifikke felt for et gitt skip (MMSI) i MySQL-databasen
    OG setter ALLTID 'last_updated'-feltet til nåværende tidspunkt ved oppdatering.
    Logger SQL-lignende info for IMO og DRAUGHT oppdateringer.
    """
    conn = get_db_connection()
    if not conn:
        return False

    cursor = None
    updated = False
    try:
        cursor = conn.cursor()

        update_fields = list(update_data.keys())
        update_values_list = list(update_data.values())

        # ---------- START MODIFISERT LOGGING FOR IMO OG DRAUGHT ----------
        specific_updates_log = []
        if "imo" in update_data:
            imo_val_to_log = update_data["imo"] if update_data["imo"] is not None else 'NULL'
            specific_updates_log.append(f"IMO = {imo_val_to_log}")
        if "draught" in update_data:
            draught_val_to_log = update_data["draught"] # Dette vil være et Decimal-objekt eller None
            # Formater Decimal for penere logging, eller vis 'NULL'
            draught_str_to_log = f"{draught_val_to_log:.2f}m" if draught_val_to_log is not None else 'NULL'
            specific_updates_log.append(f"DRAUGHT = {draught_str_to_log}")

        if specific_updates_log:
            print(f"   📝 Forbereder SQL-oppdatering for MMSI {mmsi} med: {', '.join(specific_updates_log)}")
        # ---------- SLUTT MODIFISERT LOGGING FOR IMO OG DRAUGHT ----------

        set_parts = [f"`{key}` = %s" for key in update_fields]
        set_parts.append("`last_updated` = NOW()")
        set_clause = ", ".join(set_parts)

        final_query_values = list(update_values_list) 
        final_query_values.append(mmsi)

        sql = f"UPDATE `{SHIP_TABLE}` SET {set_clause} WHERE `mmsi` = %s"

        if update_fields:
            update_log_str = ', '.join([f'`{k}`=%s' for k in update_fields])
            print(f"   💾 Utfører SQL for MMSI {mmsi}: UPDATE SET {update_log_str}, `last_updated`=NOW() WHERE `mmsi`=%s")
        else:
            print(f"   💾 Utfører SQL for MMSI {mmsi}: UPDATE SET `last_updated`=NOW() WHERE `mmsi`=%s")
        # print(f"      Verdier som sendes: {final_query_values}") # For feilsøking

        cursor.execute(sql, tuple(final_query_values))
        conn.commit()

        if cursor.rowcount > 0:
            if update_fields:
                print(f"   ✅ Rad for MMSI {mmsi} oppdatert (inkl. last_updated).")
            else:
                print(f"   ✅ Tidsstempel 'last_updated' satt for MMSI {mmsi}.")
            updated = True
        else:
            print(f"   ℹ️ Ingen rader endret for MMSI {mmsi} (UPDATE utført, men verdier var kanskje like, eller MMSI fantes ikke?).")
            # Hvis update_fields hadde innhold, men rowcount er 0, ble ingen faktiske datafelter endret.
            if update_fields:
                updated = False 
            # Hvis update_fields var tom (kun last_updated), og rowcount = 0, betyr det at MMSI ikke fantes.
            # I dette tilfellet forblir updated False, som er korrekt.

    except MySQLError as e:
        print(f"🚨 Databasefeil ved oppdatering av MMSI {mmsi} i '{SHIP_TABLE}': {e}")
        try:
             if conn and conn.is_connected():
                 conn.rollback()
                 print("   ⏪ Rollback utført.")
        except MySQLError as rb_err:
             print(f"   ⚠️ Feil under rollback: {rb_err}")
        updated = False
    finally:
        if cursor:
            cursor.close()
        if conn and conn.is_connected():
            conn.close()
    return updated

def process_ship_data():
    """
    Går gjennom skip i databasen, henter ny info fra ulike kilder
    og oppdaterer databasen. Prioriterer raske API-er, bruker Selenium kun for bilder.
    """
    global search_counter
    all_ships = get_ships_to_process()
    if not all_ships:
        print(f"ℹ️ Ingen skip funnet i databasen '{DB_NAME}.{SHIP_TABLE}' for behandling.")
        return

    # Initialiser blokkeringsflaggene
    brave_blocked_this_run = START_WITH_BRAVE_BLOCKED
    ddg_blocked_this_run = START_WITH_DDG_BLOCKED
    
    print(f"\n--- [{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] Starter behandling av {len(all_ships)} skip ---")

    for row_data in all_ships:
        mmsi = row_data.get("mmsi")
        if not mmsi or not is_valid_mmsi(str(mmsi)):
            print(f"\n--- !! Hopper over rad (DB data): Ugyldig MMSI ('{mmsi}') ---")
            continue

        mmsi = str(mmsi).strip()
        print(f"\n--- [{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] Behandler MMSI {mmsi} ---")
        
        # Hent eksisterende data fra databasen
        ship_name = row_data.get("ship_name")
        callsign = row_data.get("callsign")
        image_url = row_data.get("image_url")
        
        current_updates = {}
        search_result_data = None
        
        # STEG 1: Hent primærdata fra raske API-er
        # Prøv Kystverket først
        search_result_data = search_kystverket(mmsi)
        controlled_pause(min_delay_sec=5, max_delay_sec=10)

        # Hvis Kystverket ikke ga resultat, prøv AISHub
        if not search_result_data:
            print(f"  🔄 Kystverket ga ikke data, prøver AISHub API...")
            search_result_data = search_aishub_api(mmsi)
            controlled_pause(min_delay_sec=70, max_delay_sec=79)

        # STEG 2: Behandle primærdata og sjekk om vi mangler bilde
        if search_result_data:
            # Slå sammen funnet data med det vi har fra før
            for key, value in search_result_data.items():
                if value is not None and (row_data.get(key) is None or row_data.get(key) == ""):
                    current_updates[key] = value
            
            # Oppdater den lokale variabelen for image_url for neste sjekk
            if search_result_data.get("image_url"):
                image_url = search_result_data.get("image_url")

        # STEG 3: Hvis vi fortsatt mangler bilde, kjør bilde-søk
        if image_url is None or image_url == "":
            print(f"  📸 Mangler bilde, prøver dedikert bildesøk med Selenium...")
            image_result = search_kystverket_image(mmsi) # Den nye Selenium-funksjonen
            if image_result and image_result.get("image_url"):
                current_updates["image_url"] = image_result["image_url"]
                print(f"    -> Fant bilde-URL: {image_result['image_url']}")
            else:
                # Du kan legge inn Brave/DuckDuckGo her som en siste utvei hvis du vil
                print(f"  ℹ️ Fant heller ikke bilde med Selenium.")

        # STEG 4: Lagre all funnet informasjon i databasen
        if current_updates:
            print(f"  💾 Funnet {len(current_updates)} dataoppdatering(er) for MMSI {mmsi}: {list(current_updates.keys())}")
            update_ship_data(mmsi, current_updates)
        else:
            print(f"  -> Ingen nye data funnet for MMSI {mmsi}, setter kun tidsstempel.")
            update_ship_data(mmsi, {}) # Oppdaterer kun 'last_updated'

    print(f"\n--- [{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] Ferdig med behandling ---")


# --------------------------------------------------
# HOVEDKJØRING (ENKELTRUNDE MED MySQL)
# --------------------------------------------------
if __name__ == "__main__":
    # <--- START NYTT FOR STARTTID FOR SKRIPTET ---
    script_start_datetime_obj = datetime.now()
    script_start_time_formatted = script_start_datetime_obj.strftime("%Y-%m-%d %H:%M:%S")
    print(f"===== Starter kjøring (MySQL: {DB_USER}@{DB_HOST}/{DB_NAME}.{SHIP_TABLE}) =====")
    print(f"🚀 Starttidspunkt for skript: {script_start_time_formatted}")
    # <--- SLUTT NYTT FOR STARTTID FOR SKRIPTET ---
    start_time_epoch = time.time() # Beholdes for total varighetsberegning med time.time()

    try:
        initialize_database()
        process_ship_data()

        end_time_epoch = time.time()
        elapsed_minutes = (end_time_epoch - start_time_epoch) / 60
        script_end_time_formatted = datetime.now().strftime("%Y-%m-%d %H:%M:%S") # <--- NYTT
        print(f"\n✅ [{script_end_time_formatted}] Scriptet fullførte behandlingen på {elapsed_minutes:.2f} minutter.")
        print(f"   MySQL Databasen '{DB_NAME}.{SHIP_TABLE}' er oppdatert.")

    except MySQLError as db_err:
        end_time_epoch = time.time(); elapsed_minutes = (end_time_epoch - start_time_epoch) / 60
        script_error_time_formatted = datetime.now().strftime("%Y-%m-%d %H:%M:%S") # <--- NYTT
        print(f"\n💥 [{script_error_time_formatted}] Scriptet stoppet på grunn av en DATABASEFEIL etter {elapsed_minutes:.2f} minutter!")
        import traceback; traceback.print_exc()
        print(f"\n   Feilmelding: {db_err}")
        sys.exit(1)
    except Exception as e:
        end_time_epoch = time.time(); elapsed_minutes = (end_time_epoch - start_time_epoch) / 60
        script_error_time_formatted = datetime.now().strftime("%Y-%m-%d %H:%M:%S") # <--- NYTT
        print(f"\n💥 [{script_error_time_formatted}] Scriptet stoppet på grunn av en UVENTET FEIL etter {elapsed_minutes:.2f} minutter!")
        import traceback; traceback.print_exc()
        print(f"\n   Feilmelding: {e}")
        sys.exit(1)

    final_completion_time = datetime.now().strftime('%Y-%m-%d %H:%M:%S') # <--- NYTT
    print(f"===== Kjøring fullført [{final_completion_time}] =====")
    sys.exit(0)