# health_check.py
import mysql.connector
import subprocess
import os
from datetime import datetime, timedelta, timezone
import sys

try:
    # Forsøker å importere 'dotenv'-pakken
    from dotenv import load_dotenv
except ModuleNotFoundError:
    # Hvis pakken ikke finnes, prøver vi å installere den
    print("ADVARSEL: Pakken 'python-dotenv' er ikke installert. Prøver å installere den nå...")
    package_to_install = 'python-dotenv'
    try:
        # Bruker subprocess for å kjøre pip på en sikker måte
        # Lagt til --break-system-packages for å overstyre OS-beskyttelsen
        subprocess.check_call([sys.executable, "-m", "pip", "install", package_to_install, "--break-system-packages"])
        print(f"SUKSESS: Installasjon av '{package_to_install}' var vellykket.")
        # Importerer pakken på nytt etter installasjon
        from dotenv import load_dotenv
    except (subprocess.CalledProcessError, FileNotFoundError):
        # Håndterer feil hvis installasjonen mislykkes
        print(f"KRITISK FEIL: Kunne ikke installere '{package_to_install}'.")
        print("   Sjekk at 'pip' er tilgjengelig, og at du har internett-tilkobling og rettigheter.")
        sys.exit(1) # Avslutter skriptet, da det ikke kan fortsette

env_path = '/home/pi/key_info.env'
load_dotenv(dotenv_path=env_path)

# --- DATABASE-KONFIGURASJON (samme som de andre scriptene) ---
DB_HOST = os.getenv("DB_HOST")
DB_USER = os.getenv("DB_USER")
DB_PASSWORD = os.getenv("DB_PASSWORD")
DB_NAME = "totlando_aisdata"
LIVE_AIS_TABLE = "live_ais_data"

# Navnet på prosessen i PM2 som vi skal restarte
#PM2_PROCESS_NAME = "ais-collector"
PM2_PROCESS_NAME = "all"

# Tidsgrense i minutter. Hvis siste oppføring er eldre enn dette, restartes prosessen.
# Endre 4 timer til minutter (4 * 60 = 240)
MAX_AGE_MINUTES = 15

def check_database_freshness():
    """
    Sjekker tidsstempelet på den siste oppføringen i live_ais_data.
    Returnerer True hvis dataen er fersk nok, False ellers.
    """
    db_conn = None
    try:
        db_conn = mysql.connector.connect(
            host=DB_HOST, user=DB_USER, password=DB_PASSWORD, database=DB_NAME
        )
        with db_conn.cursor() as cursor:
            # Hent tidsstempelet for den aller siste oppføringen
            query = f"SELECT MAX(last_update_time) AS latest_update FROM {LIVE_AIS_TABLE};"
            cursor.execute(query)
            result = cursor.fetchone()

            if not result or not result[0]:
                print(f"[{datetime.now()}] ADVARSEL: Ingen data funnet i '{LIVE_AIS_TABLE}'. Kan ikke sjekke ferskhet.")
                return False # Behandler som en feil, slik at den restartes

            latest_update_time = result[0]
            
            # Gjør om databasetiden til "timezone-aware" (UTC) for korrekt sammenligning
            latest_update_time_utc = latest_update_time.replace(tzinfo=timezone.utc)
            
            # Hent nåværende tid i UTC
            now_utc = datetime.now(timezone.utc)
            
            time_difference = now_utc - latest_update_time_utc
            
            print(f"[{datetime.now()}] INFO: Siste databaseoppdatering var for {time_difference.total_seconds() / 60:.1f} minutter siden.")

            # Sjekk om differansen er større enn tillatt grense
            if time_difference > timedelta(minutes=MAX_AGE_MINUTES):
                print(f"[{datetime.now()}] ALARM: Data er for gammel! (Grense: {MAX_AGE_MINUTES} minutter).")
                return False
            else:
                print(f"[{datetime.now()}] STATUS: OK. Data er fersk nok.")
                return True

    except mysql.connector.Error as err:
        print(f"[{datetime.now()}] FEIL: Kunne ikke koble til eller sjekke databasen: {err}")
        return False # Anta at noe er galt hvis vi ikke får sjekket
    finally:
        if db_conn and db_conn.is_connected():
            db_conn.close()

def restart_pm2_process():
    """Bruker subprocess for å kjøre 'pm2 restart'."""
    command = ["/usr/local/bin/pm2", "restart", PM2_PROCESS_NAME]

    try:
        print(f"[{datetime.now()}] HANDLING: Prøver å restarte PM2-prosessen '{PM2_PROCESS_NAME}'...")
        # Bruker subprocess til å kjøre kommandoen
        result = subprocess.run(command, capture_output=True, text=True, check=True)
        print(f"[{datetime.now()}] SUKSESS: Kommandoen '{' '.join(command)}' ble utført.")
        print("--- STDOUT ---\n", result.stdout)
        
    except FileNotFoundError:
        print(f"[{datetime.now()}] KRITISK FEIL: 'pm2'-kommandoen ble ikke funnet. Er PM2 installert og i systemets PATH?")
    except subprocess.CalledProcessError as e:
        print(f"[{datetime.now()}] FEIL: 'pm2 restart' feilet med kode {e.returncode}.")
        print("--- STDERR ---\n", e.stderr)
    except Exception as e:
        print(f"[{datetime.now()}] En uventet feil oppstod under restart: {e}")

# --- HOVEDLOGIKK ---
if __name__ == "__main__":
    is_fresh = check_database_freshness()
    
    if not is_fresh:
        restart_pm2_process()
 