# Filnavn: ship_alert.py
# Versjon: 48.0 (Logger alle relevante skip ved deteksjon)
import sys
import os
import json
import time
from pathlib import Path
import math
from datetime import datetime
import logging

# --- Importer og sjekk for smart controller ---
try:
    from smart_display_controller import notify_and_display
except ImportError:
    print("FEIL: Finner ikke 'smart_display_controller.py'.")
    print("Sørg for at begge filene ligger i samme mappe.")
    sys.exit(1)

# --- Verifiser basis-pakker ---
try:
    import requests
    from haversine import haversine, Unit
    import pytz
except ImportError as e:
    print(f"FEIL: Mangler basis-pakker: '{e}'")
    print("Kjør 'smart_display_controller.py' først for å installere pakker.")
    sys.exit(1)

# --- Konfigurasjon ---
SHIP_ALERT_CONFIG = {
    "DEBUG_MODE": False, # Skrudd av for normal drift
    "api_url": "http://127.0.0.1:5000/api/live_ais_data",
    "website_to_show": "http://89.10.213.234:6380/",
    "log_file_path": "/home/pi/ais/passage.log",
    "min_draught_alert": 0.0,
    "timezone": "Europe/Oslo",
    "alert_zones": {
        "Indre Sone": [[59.83882095853034, 10.5099888449554], [59.77202830810234, 10.59253523611673], [59.80641392017936, 10.60936045556268], [59.83882095853034, 10.5099888449554]],
        "Ytre Sone": [
        [59.75097572196535, 10.58104717355498],
        [59.86801256512301, 10.6549597670859],
        [59.86623765226894, 10.55660352383354],
        [59.83907949283067, 10.50930684096248],
        [59.78080171698669, 10.50113722698391],
        [59.75097572196535, 10.58104717355498]]
    },
    "valid_course_ranges": { "inbound": (330, 30), "outbound": (150, 210) },
    "warning_time_minutes": 2,
    "check_interval_seconds": 15, # Anbefalt for raskere deteksjon
    "missing_ship_timeout_seconds": 1200
}

# --- Logging ---
passage_logger = logging.getLogger('passage_logger')
passage_logger.setLevel(logging.INFO)
file_handler = logging.FileHandler(SHIP_ALERT_CONFIG["log_file_path"])
formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s', datefmt='%Y-%m-%d %H:%M:%S')
file_handler.setFormatter(formatter)
passage_logger.addHandler(file_handler)

def log(message, level='info'):
    if SHIP_ALERT_CONFIG["DEBUG_MODE"] or level != 'debug':
        tag = f"[{level.upper()}]".ljust(8)
        print(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] {tag} {message}")

def run_startup_test(website_url):
    log("--- STARTER LYD- OG SKJERMTEST ---", level='info')
    success = notify_and_display(
        message_text="Lyd- og skjermtest for varslingssystemet er vellykket.",
        display_url=website_url,
        display_duration_seconds=15
    )
    if success: log("Test fullført. Systemet fortsetter...", level='info')
    else: log("Testen feilet.", level='error')
    log("="*70 + "\n", level='info')
    time.sleep(2)

# --- Hjelpefunksjoner ---
def is_course_valid(course_deg, valid_ranges):
    if course_deg is None: return None
    for direction, (start, end) in valid_ranges.items():
        #if direction != 'inbound': continue
        if start > end:
            if course_deg >= start or course_deg <= end: return direction
        else:
            if start <= course_deg <= end: return direction
    return None

def is_point_in_polygon(point, polygon):
    lat, lon = point; n = len(polygon); inside = False
    p1lat, p1lon = polygon[0]
    for i in range(n + 1):
        p2lat, p2lon = polygon[i % n]
        if min(p1lat, p2lat) < lat <= max(p1lat, p2lat) and lon <= max(p1lon, p2lon):
            if p1lat != p2lat: lon_intersection = (lat - p1lat) * (p2lon - p1lon) / (p2lat - p1lat) + p1lon
            if p1lon == p2lon or lon <= lon_intersection: inside = not inside
        p1lat, p1lon = p2lat, p2lon
    return inside

def get_distance_to_zone_km(ship_pos, zone_polygon):
    center_lat = sum(p[0] for p in zone_polygon) / len(zone_polygon); center_lon = sum(p[1] for p in zone_polygon) / len(zone_polygon)
    return haversine(ship_pos, (center_lat, center_lon), unit=Unit.KILOMETERS)

# --- Hovedprogram ---
def run_alert_loop():
    log("--- Varslingssystemet er aktivt ---", level='info'); ships_to_track = {}; notified_ships = {}; loop_count = 0
    while True:
        try:
            loop_count += 1
            log(f"--- Sjekk #{loop_count} | Sporede skip: {len(ships_to_track)} ---", level='debug')
            response = requests.get(SHIP_ALERT_CONFIG["api_url"], timeout=10)
            if response.status_code == 200:
                live_data = response.json().get("ships", [])
                current_time = time.time()
                
                for ship in live_data:
                    mmsi = ship.get('mmsi')
                    if not mmsi: continue
                    if mmsi in ships_to_track: continue
                    
                    ship_name = ship.get('ship_name') or f"MMSI {mmsi}"
                    draught = ship.get('draught') or 0.0
                    if not (draught >= SHIP_ALERT_CONFIG["min_draught_alert"]): continue
                    
                    lat, lon = ship.get('latitude'), ship.get('longitude')
                    if lat is None or lon is None: continue
                    if not is_point_in_polygon((lat, lon), SHIP_ALERT_CONFIG["alert_zones"]["Ytre Sone"]): continue
                    
                    course = ship.get('course_deg')
                    direction = is_course_valid(course, SHIP_ALERT_CONFIG["valid_course_ranges"])
                    if not direction: continue

                    # ==========================================================
                    # === HER ER DEN NYE LOGGINGEN =============================
                    # ==========================================================
                    # Logger til fil med en gang et relevant skip oppdages.
                    log_message = f"Skip detektert: {ship_name} (MMSI: {mmsi}), kurs: {course}°, dybde: {draught}m."
                    passage_logger.info(log_message)
                    # ==========================================================
                    
                    log(f"Nytt skip detektert ({direction}): {ship_name}.", level='info')
                    ships_to_track[mmsi] = {'name': ship_name, 'last_update': current_time, 'eta': 999, 'direction': direction}

                tracked_mmsi_list = list(ships_to_track.keys())
                if not tracked_mmsi_list:
                    time.sleep(SHIP_ALERT_CONFIG["check_interval_seconds"]); continue
                
                relevant_ships = {s['mmsi']: s for s in live_data if s.get('mmsi') in tracked_mmsi_list}
                
                for mmsi in tracked_mmsi_list:
                    ship_name_for_log = ships_to_track[mmsi].get('name', f"MMSI {mmsi}")
                    eta_minutes = 999
                    if mmsi in relevant_ships:
                        ship = relevant_ships[mmsi]; ships_to_track[mmsi]['last_update'] = current_time
                        lat, lon, speed_knots = ship.get('latitude'), ship.get('longitude'), ship.get('speed_knots')
                        if lat is not None and lon is not None and speed_knots and speed_knots > 0.5:
                            speed_kph = speed_knots * 1.852
                            distance_km = get_distance_to_zone_km((lat, lon), SHIP_ALERT_CONFIG["alert_zones"]["Indre Sone"])
                            eta_minutes = (distance_km / speed_kph) * 60
                            ships_to_track[mmsi]['eta'] = eta_minutes
                    else:
                        time_since_last_seen = current_time - ships_to_track[mmsi]['last_update']
                        if time_since_last_seen > SHIP_ALERT_CONFIG['missing_ship_timeout_seconds']:
                            del ships_to_track[mmsi]; continue
                        previous_eta = ships_to_track[mmsi].get('eta', 999)
                        time_since_last_check_minutes = SHIP_ALERT_CONFIG['check_interval_seconds'] / 60.0
                        eta_minutes = max(0, previous_eta - time_since_last_check_minutes)
                        ships_to_track[mmsi]['eta'] = eta_minutes
                    
                    if eta_minutes <= SHIP_ALERT_CONFIG["warning_time_minutes"] and time.time() - notified_ships.get(mmsi, 0) > 3600:
                        log(f"ALARM: {ship_name_for_log} vil entre Indre Sone om ca. {round(eta_minutes)} minutter.", level='info')
                        
                        # Logger også alarmen for tydelighetens skyld
                        passage_logger.warning(f"ALARM UTLØST for {ship_name_for_log} (MMSI: {mmsi}). ETA: {round(eta_minutes, 1)} min.")
                        
                        alert_message = f"Varsel: Større skip, {ship_name_for_log}, ankommer Synsvinkel Bjerkøya om cirka {round(eta_minutes)} minutter."
                        notify_and_display(message_text=alert_message, display_url=SHIP_ALERT_CONFIG["website_to_show"], display_duration_seconds=600)
                        notified_ships[mmsi] = time.time(); del ships_to_track[mmsi]

        except requests.exceptions.RequestException as e:
            log(f"En nettverksfeil oppstod: {e}", level='error')
        except Exception as e:
            log(f"En uventet feil oppstod i hovedløkken: {e}", level='error')
        
        time.sleep(SHIP_ALERT_CONFIG["check_interval_seconds"])

if __name__ == "__main__":
    run_startup_test(SHIP_ALERT_CONFIG["website_to_show"])
    run_alert_loop()

