# Filnavn: upload_via_ftp.py
# Versjon: 4.0 (Lager komprimert fil og rydder opp i gamle mapper)
import os
import ftplib
import subprocess
from datetime import datetime, timedelta
from pathlib import Path
from dotenv import load_dotenv

# --- Konfigurasjon ---
try:
    env_path = '/home/pi/key_info.env'
    load_dotenv(dotenv_path=env_path)
except Exception as e:
    print(f"ADVARSEL: Kunne ikke laste .env-fil: {e}")

FTP_HOST = os.getenv("FTP_HOST")
FTP_USER = os.getenv("FTP_USER")
FTP_PASSWORD = os.getenv("FTP_PASSWORD")

PATHS_TO_COMPRESS = [
    '/home/pi/ais/',
    '/home/pi/key_info.env',
    '/var/www/html/',
    '/home/pi/full_ais_installation.sh'
]

LOCAL_TAR_FILENAME = "ais_project.tar.gz"
LOCAL_TAR_FILEPATH = f"/tmp/{LOCAL_TAR_FILENAME}"

REMOTE_BASE_DIR = '/radio_backup/'
FOLDER_PREFIX = 'rsp_backup_'
WEEKS_TO_KEEP = 10
# --------------------

def create_tar_archive():
    """Lager en .tar.gz-fil fra de spesifiserte stiene."""
    print(f"\n--- Starter komprimering til filen: {LOCAL_TAR_FILENAME} ---")
    if os.path.exists(LOCAL_TAR_FILEPATH):
        os.remove(LOCAL_TAR_FILEPATH)

    command = ['tar', '-czvf', LOCAL_TAR_FILEPATH] + PATHS_TO_COMPRESS
    
    try:
        subprocess.run(command, check=True, capture_output=True, text=True)
        print("   -> Komprimering vellykket.")
        return True
    except subprocess.CalledProcessError as e:
        print(f"   -> FEIL under komprimering: {e.stderr}")
        return False
    except Exception as e:
        print(f"   -> En uventet feil oppstod under komprimering: {e}")
        return False

def delete_folder_recursively(ftp, path):
    """Sletter en mappe og alt innhold rekursivt på en FTP-server."""
    try:
        items = ftp.nlst(path)
    except ftplib.error_perm:
        # Mappen finnes ikke, ingenting å slette.
        print(f"   -> Mappen '{path}' finnes ikke, ingenting å slette.")
        return

    for item in items:
        # Hopp over '.' og '..' for å unngå evig løkke
        if os.path.basename(item) in ('.', '..'):
            continue
        try:
            # Sjekk om det er en mappe
            ftp.cwd(item)
            ftp.cwd('..') # Gå tilbake
            delete_folder_recursively(ftp, item)
        except ftplib.error_perm:
            # Hvis det ikke er en mappe, er det en fil
            print(f"     - Sletter fil: {item}")
            ftp.delete(item)
    
    # Når mappen er tom, slett den
    print(f"   -> Sletter tom mappe: {path}")
    ftp.rmd(path)

def main():
    """Hovedfunksjon som styrer backup-logikken."""
    if not create_tar_archive():
        print("\nBackup avbrutt på grunn av feil under komprimering.")
        return

    if not all([FTP_HOST, FTP_USER, FTP_PASSWORD]):
        print("FEIL: Mangler FTP-informasjon i .env-filen.")
        return

    # Beregn mappe-navn
    today = datetime.today()
    current_year, current_week, _ = today.isocalendar()
    date_to_delete = today - timedelta(weeks=WEEKS_TO_KEEP)
    delete_year, delete_week, _ = date_to_delete.isocalendar()

    new_weekly_folder = f"{FOLDER_PREFIX}W{current_week:02d}_{current_year}"
    old_weekly_folder = f"{FOLDER_PREFIX}W{delete_week:02d}_{delete_year}"
    
    remote_new_path = os.path.join(REMOTE_BASE_DIR, new_weekly_folder)
    remote_old_path = os.path.join(REMOTE_BASE_DIR, old_weekly_folder)

    try:
        print(f"\n--- Starter opplasting til FTP-server: {FTP_HOST} ---")
        with ftplib.FTP(FTP_HOST, FTP_USER, FTP_PASSWORD) as ftp:
            print("   -> Vellykket tilkobling.")
            
            # Opprett ny mappe
            try:
                ftp.mkd(remote_new_path)
                print(f"   -> Opprettet mappe: {remote_new_path}")
            except ftplib.error_perm:
                print(f"   -> Mappen '{remote_new_path}' finnes allerede.")
            
            ftp.cwd(remote_new_path)

            # Last opp filen
            with open(LOCAL_TAR_FILEPATH, 'rb') as file:
                print(f"   -> Laster opp {LOCAL_TAR_FILENAME}...")
                ftp.storbinary(f'STOR {LOCAL_TAR_FILENAME}', file)
            print("   -> Opplasting fullført!")

            # Slett den gamle mappen
            print(f"\n--- Starter opprydding av gammel backup ---")
            print(f"   -> Prøver å slette mappen: {remote_old_path}")
            delete_folder_recursively(ftp, remote_old_path)

    except ftplib.all_errors as e:
        print(f"   -> En alvorlig FTP-feil oppstod: {e}")
    finally:
        if os.path.exists(LOCAL_TAR_FILEPATH):
            os.remove(LOCAL_TAR_FILEPATH)
            print(f"\n   -> Slettet lokal, midlertidig fil.")

if __name__ == '__main__':
    main()
    print("\nBackup-script fullført.")
